// This file is part of Blend2D project <https://blend2d.com>
//
// See LICENSE.md for license and copyright information
// SPDX-License-Identifier: Zlib

#include "shape_data.h"

/*
// The following code can be used to dump any BLPath to a data compatible with shape data.
static void dumpShapeData(const BLPath& p) noexcept {
  BLString command_string;
  BLString vector_string;

  BLPathView view = p.view();
  size_t size = view.size;
  const uint8_t* command_data = view.command_data;
  const BLPoint* vertex_data = view.vertex_data;

  size_t i = 0;
  while (i < size) {
    uint8_t cmd = command_data[i];

    switch (cmd) {
      case BL_PATH_CMD_MOVE:
      case BL_PATH_CMD_ON:
        command_string.append(cmd == BL_PATH_CMD_MOVE ? 'M' : 'L');

        if (!vector_string.empty()) {
          vector_string.append(",\n");
        }

        vector_string.append_format("  %f, %f",
          vertex_data[i].x, vertex_data[i].y);

        i += 1;
        break;

      case BL_PATH_CMD_QUAD:
        command_string.append('Q');

        if (!vector_string.empty()) {
          vector_string.append(",\n");
        }

        vector_string.append_format("  %f, %f, %f, %f",
          vertex_data[i + 0].x, vertex_data[i + 0].y,
          vertex_data[i + 1].x, vertex_data[i + 1].y);

        i += 2;
        break;

      case BL_PATH_CMD_CUBIC:
        if (!vector_string.empty()) {
          vector_string.append(",\n");
        }

        command_string.append('C');
        vector_string.append_format("  %f, %f, %f, %f, %f, %f",
          vertex_data[i + 0].x, vertex_data[i + 0].y,
          vertex_data[i + 1].x, vertex_data[i + 1].y,
          vertex_data[i + 2].x, vertex_data[i + 2].y);

        i += 3;
        break;

      case BL_PATH_CMD_CLOSE:
        command_string.append("Z");

        i += 1;
        break;
    }
  }

  BLString data;
  data.append_format("constexpr char commands[] =\n  \"%s\";\n", command_string.data());
  data.append_format("constexpr double vertices[] = {\n%s\n};\n", vector_string.data());

  printf("%s\n", data.data());
}
*/

#define ARRAY_SIZE(X) uint32_t(sizeof(X) / sizeof(X[0]))

namespace blbench {

// Public domain.
constexpr char butterfly_commands[] =
  "MCCCCCCLCCCCCCCCCCCCCCCCCCLLCCCCCCCCCCCCCCLLLCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCLCCLCCCCCCCCCCCCCCCCCCCCCCCCCCCCLCCCCCCCCCCCCCCLLCCCCCCCCCCCCCCCCCCCCZ";

constexpr double butterfly_vertices[] = {
  0.996537, 0.064954,
  0.993752, 0.050263, 0.989116, 0.037814, 0.982760, 0.027953,
  0.976193, 0.017700, 0.966729, 0.009674, 0.956111, 0.005354,
  0.947852, 0.001955, 0.938374, 0.000230, 0.927941, 0.000230,
  0.924588, 0.000230, 0.921051, 0.000407, 0.917431, 0.000756,
  0.903445, 0.002160, 0.888782, 0.005829, 0.872604, 0.011972,
  0.842838, 0.023344, 0.814658, 0.040444, 0.787406, 0.056982,
  0.787049, 0.057198,
  0.779094, 0.062111, 0.770959, 0.067003, 0.763092, 0.071733,
  0.755616, 0.076228, 0.747885, 0.080877, 0.740322, 0.085536,
  0.715897, 0.100667, 0.687906, 0.118856, 0.664830, 0.143282,
  0.655450, 0.153337, 0.646158, 0.164412, 0.637213, 0.176198,
  0.628832, 0.187242, 0.620487, 0.199262, 0.612411, 0.211927,
  0.599025, 0.233035, 0.585902, 0.256540, 0.572296, 0.283785,
  0.559363, 0.309680, 0.547303, 0.336361, 0.537211, 0.359044,
  0.531569, 0.371160, 0.524734, 0.385013, 0.514991, 0.396163,
  0.515458, 0.393140, 0.516021, 0.390068, 0.516567, 0.387096,
  0.518222, 0.378075, 0.519933, 0.368746, 0.519220, 0.359351,
  0.518849, 0.353698, 0.517257, 0.347795, 0.514490, 0.341807,
  0.525767, 0.316394, 0.535723, 0.289763, 0.545350, 0.264009,
  0.549268, 0.253532, 0.553317, 0.242697, 0.557400, 0.232097,
  0.558269, 0.229903, 0.559129, 0.227657, 0.559961, 0.225484,
  0.562894, 0.217827, 0.565926, 0.209908, 0.569931, 0.202734,
  0.570101, 0.202412, 0.570272, 0.202089, 0.570443, 0.201765,
  0.573622, 0.195744, 0.576909, 0.189518, 0.577757, 0.182438,
  0.578339, 0.177538, 0.576559, 0.172383, 0.572995, 0.168649,
  0.572952, 0.168603,
  0.572927, 0.168664,
  0.571876, 0.171135, 0.571043, 0.173762, 0.570237, 0.176303,
  0.569698, 0.178001, 0.569141, 0.179758, 0.568523, 0.181453,
  0.559856, 0.207043, 0.549627, 0.232362, 0.539734, 0.256848,
  0.533800, 0.271536, 0.527664, 0.286723, 0.521930, 0.301809,
  0.520066, 0.306464, 0.518896, 0.311494, 0.517766, 0.316358,
  0.516331, 0.322533, 0.514848, 0.328918, 0.511928, 0.334565,
  0.509243, 0.339707, 0.505293, 0.343102, 0.501099, 0.343881,
  0.496390, 0.343827, 0.491623, 0.340706, 0.488658, 0.335735,
  0.485246, 0.330155, 0.483362, 0.323648, 0.481539, 0.317355,
  0.480411, 0.313464, 0.479246, 0.309440, 0.477730, 0.305670,
  0.473688, 0.295601, 0.469520, 0.285535, 0.465598, 0.276088,
  0.460318, 0.263417, 0.455498, 0.249775, 0.450860, 0.234382,
  0.450381, 0.232850, 0.449904, 0.231314, 0.449427, 0.229777,
  0.442988, 0.209044, 0.436330, 0.187605, 0.424264, 0.169941,
  0.424243, 0.169910,
  0.422251, 0.170903,
  0.422260, 0.170944,
  0.427099, 0.192915, 0.434511, 0.213736, 0.441800, 0.232965,
  0.446725, 0.245803, 0.451991, 0.258730, 0.457084, 0.271232,
  0.466770, 0.295008, 0.476785, 0.319592, 0.484880, 0.344614,
  0.484479, 0.345589, 0.483947, 0.346506, 0.483432, 0.347394,
  0.482759, 0.348553, 0.482064, 0.349751, 0.481652, 0.351096,
  0.478890, 0.358331, 0.479848, 0.366528, 0.481295, 0.374308,
  0.481535, 0.375769, 0.481819, 0.377267, 0.482092, 0.378715,
  0.483208, 0.384609, 0.484361, 0.390701, 0.483123, 0.396684,
  0.464551, 0.354942, 0.444111, 0.310539, 0.420513, 0.268591,
  0.409919, 0.249696, 0.398977, 0.231782, 0.387993, 0.215345,
  0.376149, 0.197621, 0.363804, 0.180939, 0.351302, 0.165762,
  0.335506, 0.146419, 0.320183, 0.130559, 0.304457, 0.117277,
  0.285094, 0.100717, 0.263209, 0.085094, 0.233498, 0.066623,
  0.216130, 0.055916, 0.193166, 0.042107, 0.169549, 0.030252,
  0.157334, 0.024158, 0.146468, 0.019259, 0.136331, 0.015277,
  0.124194, 0.010508, 0.112940, 0.006966, 0.101924, 0.004449,
  0.089957, 0.001783, 0.077226, 0.000287, 0.064083, 0.000001,
  0.063913, 0.000000, 0.063743, 0.000000, 0.063573, 0.000000,
  0.047727, 0.000000, 0.035764, 0.003718, 0.026998, 0.011369,
  0.018384, 0.019899, 0.013592, 0.032061, 0.010042, 0.042833,
  0.004378, 0.060560, 0.001059, 0.080429, 0.000178, 0.101887,
  -0.000434, 0.118425, 0.000530, 0.133075, 0.003124, 0.146675,
  0.005473, 0.158080, 0.011228, 0.167827, 0.016793, 0.177252,
  0.019569, 0.181954, 0.022439, 0.186816, 0.024848, 0.191819,
  0.028514, 0.198869, 0.031442, 0.207183, 0.034062, 0.217987,
  0.040758, 0.245180, 0.044408, 0.273539, 0.047938, 0.300966,
  0.050632, 0.321896, 0.053418, 0.343539, 0.057539, 0.364526,
  0.059785, 0.375450, 0.061848, 0.383680, 0.064226, 0.391211,
  0.067248, 0.400780, 0.070726, 0.408921, 0.074857, 0.416099,
  0.080169, 0.425346, 0.087823, 0.433270, 0.096990, 0.439015,
  0.102601, 0.442564, 0.108760, 0.445464, 0.115819, 0.447879,
  0.121663, 0.449879, 0.128174, 0.451550, 0.135724, 0.452987,
  0.148144, 0.455266, 0.160328, 0.456346, 0.170609, 0.457103,
  0.176430, 0.457531, 0.182358, 0.457902, 0.188090, 0.458261,
  0.208442, 0.459535, 0.229488, 0.460852, 0.249919, 0.464963,
  0.250733, 0.465151, 0.251518, 0.465456, 0.252249, 0.465763,
  0.244052, 0.471806, 0.235637, 0.477802, 0.227498, 0.483601,
  0.213265, 0.493741, 0.198547, 0.504227, 0.184513, 0.515315,
  0.157191, 0.536822, 0.137272, 0.556896, 0.121828, 0.578488,
  0.116857, 0.585609, 0.111658, 0.593254, 0.107522, 0.601548,
  0.107517, 0.601571,
  0.107229, 0.617612, 0.108009, 0.634682, 0.110046, 0.656919,
  0.114472, 0.703027, 0.123678, 0.747454, 0.134364, 0.796592,
  0.134397, 0.796741,
  0.142032, 0.831679, 0.149928, 0.867806, 0.161323, 0.902046,
  0.168637, 0.923705, 0.176050, 0.940619, 0.184650, 0.955275,
  0.190128, 0.964972, 0.197358, 0.973862, 0.206137, 0.981700,
  0.215385, 0.989861, 0.223951, 0.995206, 0.232323, 0.998040,
  0.236067, 0.999347, 0.240162, 0.999983, 0.244841, 0.999983,
  0.246773, 0.999983, 0.248772, 0.999877, 0.250954, 0.999656,
  0.262942, 0.998220, 0.275491, 0.993342, 0.288254, 0.985158,
  0.301910, 0.976633, 0.314643, 0.965047, 0.326102, 0.950719,
  0.341620, 0.931379, 0.355720, 0.908251, 0.370474, 0.877935,
  0.381824, 0.854509, 0.392204, 0.829376, 0.402205, 0.801098,
  0.410327, 0.778394, 0.417695, 0.753602, 0.424103, 0.727411,
  0.426925, 0.716082, 0.429640, 0.704514, 0.432266, 0.693326,
  0.436645, 0.674668, 0.441173, 0.655375, 0.446330, 0.636628,
  0.452661, 0.613947, 0.460882, 0.587568, 0.473848, 0.563846,
  0.476793, 0.577381, 0.476673, 0.591583, 0.476556, 0.605318,
  0.476542, 0.607014, 0.476527, 0.608767, 0.476518, 0.610487,
  0.476470, 0.614319, 0.476391, 0.618218, 0.476313, 0.621989,
  0.476063, 0.634239, 0.475803, 0.646905, 0.476596, 0.659341,
  0.477128, 0.667063, 0.478080, 0.676065, 0.481106, 0.684272,
  0.483337, 0.690191, 0.486853, 0.694657, 0.491005, 0.696847,
  0.493582, 0.698189, 0.496437, 0.698870, 0.499491, 0.698869,
  0.502595, 0.698869, 0.505815, 0.698142, 0.508803, 0.696767,
  0.515676, 0.693339, 0.518814, 0.685534, 0.520073, 0.681187,
  0.522839, 0.671664, 0.523222, 0.661301, 0.523364, 0.652336,
  0.523469, 0.643817, 0.523203, 0.635163, 0.522947, 0.626793,
  0.522661, 0.617472, 0.522365, 0.607833, 0.522580, 0.598348,
  0.522939, 0.586709, 0.523665, 0.574444, 0.528129, 0.563620,
  0.540224, 0.596614, 0.550134, 0.631187, 0.559718, 0.664623,
  0.560086, 0.665910,
  0.561636, 0.671376, 0.563206, 0.676935, 0.564725, 0.682311,
  0.573685, 0.714023, 0.582949, 0.746814, 0.593021, 0.778688,
  0.602829, 0.809554, 0.616473, 0.849915, 0.635089, 0.888377,
  0.642467, 0.903459, 0.649743, 0.916506, 0.657333, 0.928262,
  0.666111, 0.941858, 0.675091, 0.953417, 0.684786, 0.963601,
  0.694124, 0.973455, 0.705544, 0.981993, 0.718730, 0.988978,
  0.730344, 0.994940, 0.740943, 0.998413, 0.751134, 0.999595,
  0.753296, 0.999864, 0.755462, 1.000000, 0.757571, 1.000000,
  0.764286, 1.000000, 0.770528, 0.998652, 0.776124, 0.995996,
  0.786561, 0.991208, 0.796201, 0.982452, 0.805598, 0.969225,
  0.812476, 0.959519, 0.818763, 0.948102, 0.825386, 0.933294,
  0.836892, 0.907222, 0.845008, 0.880016, 0.851429, 0.856842,
  0.860333, 0.823959, 0.866557, 0.798444, 0.871610, 0.774111,
  0.883378, 0.718009, 0.890229, 0.659922, 0.891971, 0.601464,
  0.891971, 0.601449,
  0.891964, 0.601436,
  0.885844, 0.590088, 0.878517, 0.578874, 0.870186, 0.568105,
  0.862500, 0.558167, 0.853755, 0.548345, 0.844194, 0.538911,
  0.831804, 0.526844, 0.820427, 0.517254, 0.809413, 0.509593,
  0.803694, 0.505538, 0.797760, 0.501591, 0.792021, 0.497774,
  0.776458, 0.487424, 0.760368, 0.476723, 0.746737, 0.462659,
  0.756830, 0.461866, 0.767107, 0.461367, 0.777048, 0.460884,
  0.784049, 0.460544, 0.791287, 0.460192, 0.798401, 0.459740,
  0.825745, 0.458022, 0.850344, 0.456095, 0.874493, 0.449883,
  0.886652, 0.446557, 0.900634, 0.441847, 0.912144, 0.432128,
  0.919904, 0.425456, 0.926036, 0.415550, 0.931439, 0.400953,
  0.937185, 0.385330, 0.940611, 0.369085, 0.943900, 0.352426,
  0.944476, 0.349326, 0.945069, 0.346176, 0.945642, 0.343130,
  0.947313, 0.334259, 0.949041, 0.325086, 0.950540, 0.316019,
  0.952343, 0.304715, 0.953756, 0.293146, 0.955122, 0.281958,
  0.955879, 0.275752, 0.956663, 0.269335, 0.957504, 0.263038,
  0.961054, 0.236741, 0.965083, 0.211233, 0.975254, 0.188238,
  0.977302, 0.183515, 0.979523, 0.178784, 0.981670, 0.174210,
  0.985841, 0.165325, 0.990154, 0.156138, 0.993408, 0.146564,
  0.997628, 0.134251, 0.999838, 0.121996, 0.999978, 0.110139,
  1.000176, 0.093057, 0.999051, 0.078277, 0.996537, 0.064954
};

// Public domain.
constexpr char fish_commands[] =
  "MCCCCCCCCCCCCCCZMCCCCCCCCCCCCCCCCCCCCCCCLLZMCCCCCCCCCCCCCCCCCCCCCCCLLCZMCCCZMCCCCCCCZMCCCCCCZMCCCZMCCCCCZMCCCCCCZMCCCCCCCZMCCCCLLCCLLCCCZMCCLLCZMCCCCLLCCZMCCCCCZMLLCCLLCCZMCCCCCZMCCCCZMCCCLLCCCCZMCLLLLCCZMCCCCCCZMCCCCCCZMCCCLLCZMCCCCCZMCCCCCCCZMCCCCCCCZMCCCCLLCZMCCCCCCCCCZMCCCCCCCZMCCCCCCZMCCCCCCCZMCCCCCZMCCCCZMCLLCCCZMCCCCCZMCCLLZMCCCCCZMCCCCZMCCCCZ";

constexpr double fish_vertices[] = {
  0.437004, 0.847854,
  0.428891, 0.844338, 0.405683, 0.812911, 0.399583, 0.797114,
  0.396960, 0.790304, 0.358929, 0.754357, 0.347127, 0.747547,
  0.325473, 0.734987, 0.300383, 0.709143, 0.289068, 0.687641,
  0.269034, 0.649600, 0.270141, 0.593301, 0.292185, 0.529343,
  0.309919, 0.477855, 0.313213, 0.483487, 0.317117, 0.572051,
  0.318520, 0.603739, 0.321814, 0.637013, 0.324650, 0.647931,
  0.333037, 0.680245, 0.372226, 0.735769, 0.372104, 0.715171,
  0.372104, 0.712883, 0.367560, 0.705331, 0.362010, 0.698420,
  0.349353, 0.682629, 0.333311, 0.652375, 0.337642, 0.652375,
  0.339411, 0.652375, 0.346181, 0.655780, 0.352708, 0.659944,
  0.359204, 0.664108, 0.378235, 0.671945, 0.395008, 0.677365,
  0.434777, 0.690198, 0.453442, 0.700904, 0.467258, 0.718744,
  0.486044, 0.743025, 0.489551, 0.770377, 0.478847, 0.808781,
  0.469514, 0.842272, 0.455180, 0.855613, 0.437004, 0.847854,
  0.915453, 0.797505,
  0.912281, 0.787960, 0.906548, 0.758822, 0.902705, 0.732755,
  0.898130, 0.701585, 0.893830, 0.683438, 0.890140, 0.679821,
  0.887029, 0.676790, 0.884498, 0.670644, 0.884498, 0.666168,
  0.884498, 0.655512, 0.862936, 0.626698, 0.841801, 0.609120,
  0.827345, 0.597091, 0.819263, 0.594780, 0.783123, 0.592291,
  0.756224, 0.590443, 0.740121, 0.587032, 0.738200, 0.582768,
  0.736523, 0.579117, 0.734296, 0.553719, 0.733198, 0.526334,
  0.732100, 0.498949, 0.727800, 0.456566, 0.723652, 0.432156,
  0.712643, 0.367366, 0.707001, 0.327919, 0.706208, 0.310191,
  0.704774, 0.277843, 0.726245, 0.265496, 0.760982, 0.278697,
  0.767691, 0.281248, 0.782818, 0.283844, 0.794621, 0.284458,
  0.806393, 0.285072, 0.818958, 0.287567, 0.822557, 0.289989,
  0.831554, 0.296091, 0.833871, 0.313741, 0.828229, 0.333640,
  0.820879, 0.359702, 0.822282, 0.365870, 0.837775, 0.375538,
  0.856836, 0.387433, 0.870591, 0.386747, 0.895599, 0.372647,
  0.923443, 0.356967, 0.955405, 0.325251, 0.974527, 0.294382,
  0.978248, 0.288343, 0.984256, 0.285128, 0.989776, 0.286183,
  0.998133, 0.287790, 0.998956, 0.289861, 0.999840, 0.312077,
  1.001213, 0.346495, 0.994076, 0.375706, 0.964981, 0.454986,
  0.936375, 0.532865, 0.935978, 0.535790, 0.945310, 0.600066,
  0.948513, 0.621959, 0.952081, 0.660022, 0.953240, 0.684649,
  0.955222, 0.726335, 0.954795, 0.731582, 0.946988, 0.760776,
  0.942383, 0.778080, 0.934697, 0.797282, 0.929909, 0.803534,
  0.921217, 0.814865,
  0.915453, 0.797505,
  0.096496, 0.729908,
  0.078710, 0.724828, 0.064104, 0.702467, 0.064104, 0.680318,
  0.064104, 0.665983, 0.064510, 0.665699, 0.084690, 0.665732,
  0.113157, 0.665782, 0.137183, 0.650979, 0.145472, 0.628283,
  0.155796, 0.600010, 0.156805, 0.579586, 0.148766, 0.561596,
  0.137991, 0.537487, 0.117756, 0.525452, 0.046553, 0.500814,
  0.024610, 0.493222, 0.017738, 0.476343, 0.003542, 0.395209,
  -0.004623, 0.348522, 0.000690, 0.334204, 0.026171, 0.334204,
  0.037965, 0.334204, 0.043777, 0.330096, 0.059154, 0.310877,
  0.092007, 0.269828, 0.168849, 0.207204, 0.198291, 0.197486,
  0.204162, 0.195543, 0.211958, 0.195995, 0.215617, 0.198496,
  0.222479, 0.203174, 0.271486, 0.339585, 0.271486, 0.354009,
  0.271486, 0.358296, 0.265255, 0.389817, 0.257637, 0.424062,
  0.250025, 0.458313, 0.244532, 0.487685, 0.245438, 0.489343,
  0.247606, 0.493312, 0.247143, 0.494819, 0.260656, 0.440261,
  0.274332, 0.385056, 0.275103, 0.383046, 0.281062, 0.387232,
  0.286101, 0.390777, 0.301981, 0.446641, 0.301981, 0.460830,
  0.301981, 0.465865, 0.297824, 0.481835, 0.292749, 0.496315,
  0.281023, 0.529756, 0.269601, 0.582958, 0.265045, 0.625347,
  0.261053, 0.662528, 0.252160, 0.680284, 0.237530, 0.680284,
  0.226731, 0.680284, 0.226636, 0.675171, 0.236581, 0.630365,
  0.240866, 0.611057, 0.243883, 0.588640, 0.243279, 0.580557,
  0.242275, 0.567083, 0.241553, 0.568188, 0.234639, 0.593764,
  0.222671, 0.638046, 0.210259, 0.675629, 0.199313, 0.700709,
  0.189185, 0.723935,
  0.162478, 0.727731,
  0.117994, 0.734150, 0.111952, 0.734318, 0.096496, 0.729908,
  0.194305, 0.589237,
  0.194864, 0.571777, 0.194089, 0.557482, 0.192588, 0.557482,
  0.188968, 0.557482, 0.186187, 0.577543, 0.186138, 0.603996,
  0.186068, 0.638420, 0.193143, 0.625643, 0.194305, 0.589237,
  0.199057, 0.461751,
  0.208481, 0.452681, 0.212702, 0.433406, 0.211274, 0.405921,
  0.209487, 0.371497, 0.198136, 0.350805, 0.178858, 0.346825,
  0.166748, 0.344330, 0.163567, 0.346138, 0.152908, 0.361595,
  0.139007, 0.381740, 0.132044, 0.411140, 0.137088, 0.428388,
  0.138857, 0.434439, 0.145460, 0.447233, 0.151761, 0.456817,
  0.162301, 0.472848, 0.164442, 0.473954, 0.178471, 0.470571,
  0.186858, 0.468539, 0.196120, 0.464570, 0.199057, 0.461751,
  0.158397, 0.449918,
  0.149824, 0.439379, 0.147808, 0.433094, 0.147061, 0.414595,
  0.146323, 0.396298, 0.147693, 0.389259, 0.154399, 0.376984,
  0.165924, 0.355890, 0.175842, 0.352574, 0.188959, 0.365429,
  0.198956, 0.375226, 0.199890, 0.378318, 0.200719, 0.404436,
  0.201811, 0.438731, 0.195867, 0.454294, 0.179743, 0.459329,
  0.171310, 0.461963, 0.166446, 0.459820, 0.158397, 0.449918,
  0.060905, 0.436074,
  0.060710, 0.407646, 0.051963, 0.376069, 0.044293, 0.376069,
  0.035500, 0.376069, 0.035461, 0.389270, 0.044137, 0.425179,
  0.053259, 0.462940, 0.061131, 0.468053, 0.060905, 0.436074,
  0.468203, 0.676422,
  0.443927, 0.666212, 0.439230, 0.662495, 0.439230, 0.653552,
  0.439261, 0.642701, 0.472686, 0.558654, 0.477352, 0.557772,
  0.480707, 0.557136, 0.493608, 0.640513, 0.493943, 0.664934,
  0.494126, 0.678733, 0.493120, 0.680217, 0.484214, 0.679637,
  0.478755, 0.679280, 0.471558, 0.677834, 0.468203, 0.676422,
  0.519622, 0.671839,
  0.518646, 0.667195, 0.519104, 0.660290, 0.520659, 0.656489,
  0.522215, 0.652687, 0.527369, 0.630745, 0.532126, 0.607719,
  0.543502, 0.552603, 0.545271, 0.548752, 0.552041, 0.564515,
  0.557562, 0.577426, 0.566131, 0.653558, 0.563051, 0.662629,
  0.561099, 0.668428, 0.537402, 0.680284, 0.527765, 0.680284,
  0.524258, 0.680284, 0.520598, 0.676483, 0.519622, 0.671839,
  0.502025, 0.651375,
  0.498548, 0.621166, 0.488393, 0.542617, 0.486563, 0.531967,
  0.485068, 0.523303, 0.489216, 0.508941, 0.503153, 0.474517,
  0.513370, 0.449276, 0.522855, 0.429862, 0.524197, 0.431380,
  0.525539, 0.432893, 0.529900, 0.451676, 0.533865, 0.473105,
  0.540971, 0.511230, 0.541001, 0.512703, 0.535969, 0.541752,
  0.527308, 0.591799, 0.511052, 0.659726, 0.507179, 0.662087,
  0.505166, 0.663321, 0.502848, 0.658504, 0.502025, 0.651375,
  0.383572, 0.653887,
  0.377289, 0.652682, 0.372135, 0.649360, 0.372135, 0.646514,
  0.372135, 0.643667, 0.378570, 0.625883, 0.386408, 0.606999,
  0.401931, 0.569667, 0.401809, 0.569667, 0.413734, 0.605659,
  0.421114, 0.627948, 0.421267, 0.606474, 0.413978, 0.571961,
  0.406292, 0.535640,
  0.419772, 0.503912,
  0.427183, 0.486468, 0.435875, 0.461238, 0.439108, 0.447852,
  0.448623, 0.408539, 0.453076, 0.415499, 0.462042, 0.483582,
  0.467349, 0.524113,
  0.452527, 0.578475,
  0.437553, 0.633469, 0.431636, 0.647524, 0.425354, 0.643120,
  0.423463, 0.641791, 0.420138, 0.644582, 0.418004, 0.649321,
  0.414191, 0.657733, 0.407970, 0.658554, 0.383572, 0.653887,
  0.585650, 0.651353,
  0.585650, 0.647306, 0.591994, 0.621657, 0.599770, 0.594350,
  0.615538, 0.538793, 0.617886, 0.539056, 0.619625, 0.596555,
  0.620692, 0.632838,
  0.610018, 0.642076,
  0.592878, 0.656907, 0.585619, 0.659670, 0.585650, 0.651353,
  0.572414, 0.635623,
  0.571590, 0.627948, 0.568724, 0.603728, 0.566070, 0.581797,
  0.563417, 0.559865, 0.561252, 0.534746, 0.561282, 0.525977,
  0.561343, 0.508969, 0.596263, 0.378860, 0.600777, 0.378860,
  0.605870, 0.378860, 0.609926, 0.415287, 0.610750, 0.468171,
  0.611543, 0.521199,
  0.597270, 0.574227,
  0.589401, 0.603393, 0.580953, 0.632274, 0.578452, 0.638414,
  0.574000, 0.649366, 0.573908, 0.649310, 0.572414, 0.635623,
  0.345602, 0.626296,
  0.342644, 0.618096, 0.340265, 0.602187, 0.340265, 0.590934,
  0.340265, 0.567473, 0.368963, 0.439530, 0.373599, 0.442198,
  0.377533, 0.444436, 0.393056, 0.517264, 0.393300, 0.534534,
  0.393605, 0.556723, 0.361949, 0.641211, 0.353318, 0.641211,
  0.352006, 0.641211, 0.348530, 0.634501, 0.345602, 0.626296,
  0.628377, 0.569455,
  0.624504, 0.507970,
  0.638930, 0.458771,
  0.655856, 0.401065, 0.657655, 0.396845, 0.662077, 0.404927,
  0.665859, 0.411866, 0.666561, 0.418006, 0.669519, 0.472128,
  0.671715, 0.512357,
  0.657350, 0.561439,
  0.649451, 0.588428, 0.640576, 0.615110, 0.637618, 0.620725,
  0.632464, 0.630510, 0.632068, 0.628378, 0.628377, 0.569455,
  0.656954, 0.610951,
  0.655307, 0.602974, 0.667933, 0.555958, 0.675740, 0.541160,
  0.679339, 0.534333, 0.680315, 0.535165, 0.682785, 0.547094,
  0.685652, 0.560797, 0.687269, 0.581328, 0.686598, 0.595444,
  0.686293, 0.602327, 0.668817, 0.618883, 0.661864, 0.618883,
  0.660095, 0.618883, 0.657869, 0.615311, 0.656954, 0.610951,
  0.695412, 0.594462,
  0.695412, 0.572229, 0.717583, 0.490532, 0.723622, 0.490510,
  0.724872, 0.490504, 0.725909, 0.511704, 0.725909, 0.537627,
  0.725909, 0.579999, 0.725238, 0.585631, 0.719139, 0.593446,
  0.710630, 0.604336, 0.695412, 0.604990, 0.695412, 0.594462,
  0.322210, 0.504972,
  0.317788, 0.482232, 0.314189, 0.460250, 0.314159, 0.456130,
  0.314128, 0.441260, 0.339411, 0.322616, 0.342278, 0.324357,
  0.343864, 0.325334, 0.350238, 0.342901, 0.356368, 0.363398,
  0.367560, 0.400663,
  0.358716, 0.446976,
  0.353836, 0.472446, 0.348865, 0.495807, 0.347676, 0.498877,
  0.346456, 0.501947, 0.343833, 0.513881, 0.341820, 0.525391,
  0.339777, 0.536907, 0.336331, 0.546323, 0.334165, 0.546323,
  0.332000, 0.546323, 0.326632, 0.527708, 0.322210, 0.504972,
  0.689495, 0.525386,
  0.686689, 0.506876, 0.687635, 0.498196, 0.697638, 0.450292,
  0.708922, 0.396136,
  0.715906, 0.426485,
  0.722921, 0.456839,
  0.709928, 0.501472,
  0.702792, 0.526016, 0.695991, 0.546156, 0.694802, 0.546217,
  0.693612, 0.546279, 0.691233, 0.536901, 0.689495, 0.525386,
  0.399125, 0.494685,
  0.392995, 0.485514, 0.381284, 0.433680, 0.381284, 0.415617,
  0.381284, 0.398882, 0.414557, 0.278373, 0.420321, 0.274243,
  0.422487, 0.272680, 0.426177, 0.276113, 0.428525, 0.281868,
  0.434289, 0.296063, 0.448410, 0.367595, 0.448410, 0.382477,
  0.448379, 0.396320, 0.411874, 0.501433, 0.407024, 0.501567,
  0.405256, 0.501618, 0.401687, 0.498525, 0.399125, 0.494685,
  0.477383, 0.480730,
  0.468538, 0.440523, 0.461341, 0.381572, 0.463811, 0.369616,
  0.465946, 0.359317, 0.485190, 0.307165, 0.493333, 0.289593,
  0.496505, 0.282733, 0.518128, 0.373797, 0.518372, 0.395086,
  0.518463, 0.402477, 0.513218, 0.421316, 0.506722, 0.436951,
  0.500226, 0.452586, 0.493303, 0.471029, 0.491351, 0.477939,
  0.487142, 0.492938, 0.480372, 0.494289, 0.477383, 0.480730,
  0.552804, 0.478079,
  0.546735, 0.464464, 0.533651, 0.398631, 0.535359, 0.390163,
  0.538805, 0.373088, 0.587205, 0.246032, 0.589340, 0.248449,
  0.590560, 0.249822, 0.594403, 0.267740, 0.597849, 0.288270,
  0.604162, 0.325597,
  0.585132, 0.397040,
  0.564698, 0.473741, 0.558171, 0.490074, 0.552804, 0.478079,
  0.677723, 0.453529,
  0.665524, 0.362253, 0.665249, 0.368098, 0.683731, 0.329002,
  0.693765, 0.307762, 0.695229, 0.306384, 0.697790, 0.315583,
  0.699376, 0.321220, 0.701511, 0.339345, 0.702517, 0.355856,
  0.704103, 0.381388, 0.702975, 0.391832, 0.695137, 0.425631,
  0.683273, 0.476683, 0.681261, 0.479954, 0.677723, 0.453529,
  0.624260, 0.464447,
  0.623132, 0.462404, 0.622217, 0.452753, 0.622217, 0.443007,
  0.622217, 0.433261, 0.620173, 0.412954, 0.617673, 0.397883,
  0.611543, 0.360852, 0.611909, 0.332289, 0.618862, 0.307467,
  0.623864, 0.289710, 0.640394, 0.255108, 0.640576, 0.262057,
  0.640637, 0.263358, 0.644572, 0.285647, 0.649360, 0.311580,
  0.659424, 0.365915, 0.659058, 0.370241, 0.639723, 0.430493,
  0.626669, 0.471168, 0.627249, 0.469912, 0.624260, 0.464447,
  0.298483, 0.416526,
  0.294027, 0.403482, 0.288254, 0.383336, 0.285646, 0.371754,
  0.280986, 0.351022, 0.281108, 0.349845, 0.293356, 0.297793,
  0.300203, 0.268694, 0.307339, 0.244938, 0.309230, 0.244988,
  0.314110, 0.245127, 0.324449, 0.266629, 0.330548, 0.289370,
  0.335733, 0.308683, 0.335702, 0.308862, 0.321704, 0.374494,
  0.313957, 0.410660, 0.307400, 0.440255, 0.307126, 0.440250,
  0.306821, 0.440244, 0.302939, 0.429571, 0.298483, 0.416526,
  0.367298, 0.372273,
  0.364431, 0.362080, 0.358393, 0.341181, 0.353879, 0.325831,
  0.349365, 0.310481, 0.346346, 0.294154, 0.347139, 0.289549,
  0.351073, 0.267237, 0.378308, 0.166746, 0.380412, 0.166768,
  0.385414, 0.166824, 0.401578, 0.199428, 0.408562, 0.223615,
  0.415850, 0.248739,
  0.398314, 0.313512,
  0.376966, 0.392418, 0.374373, 0.397330, 0.367298, 0.372273,
  0.525246, 0.360997,
  0.523233, 0.357313, 0.521586, 0.351341, 0.521586, 0.347718,
  0.521586, 0.344095, 0.518049, 0.326936, 0.513687, 0.309588,
  0.504050, 0.270966, 0.504935, 0.264854, 0.529302, 0.201460,
  0.545832, 0.158496, 0.552145, 0.149872, 0.559800, 0.159852,
  0.564985, 0.166601, 0.582582, 0.220098, 0.582582, 0.229107,
  0.582582, 0.233701, 0.577092, 0.251803, 0.570383, 0.269331,
  0.563673, 0.286864, 0.558184, 0.302460, 0.558184, 0.303989,
  0.558184, 0.309437, 0.533541, 0.367696, 0.531254, 0.367696,
  0.529973, 0.367696, 0.527259, 0.364681, 0.525246, 0.360997,
  0.446013, 0.310481,
  0.441042, 0.285150, 0.436040, 0.258870, 0.434912, 0.252077,
  0.432624, 0.238557, 0.435095, 0.227606, 0.449490, 0.187667,
  0.460926, 0.155984, 0.468490, 0.154711, 0.474315, 0.183492,
  0.476480, 0.194237, 0.481268, 0.216710, 0.484897, 0.233433,
  0.489960, 0.256604, 0.490784, 0.266774, 0.488466, 0.276185,
  0.484257, 0.293065, 0.459554, 0.356197, 0.457083, 0.356381,
  0.455955, 0.356465, 0.451014, 0.335806, 0.446013, 0.310481,
  0.658887, 0.321645,
  0.656936, 0.305524, 0.653825, 0.285150, 0.651995, 0.276370,
  0.649250, 0.263319, 0.649738, 0.256682, 0.654496, 0.240087,
  0.660260, 0.220070, 0.672855, 0.197446, 0.678192, 0.197446,
  0.680449, 0.197446, 0.692008, 0.265306, 0.693472, 0.287310,
  0.693838, 0.292680, 0.666481, 0.350950, 0.663584, 0.350950,
  0.662974, 0.350950, 0.660870, 0.337765, 0.658887, 0.321645,
  0.250217, 0.260762,
  0.236508, 0.223123, 0.226667, 0.190770, 0.228115, 0.188119,
  0.229552, 0.185490, 0.242602, 0.180684, 0.257116, 0.177430,
  0.280919, 0.172104, 0.284253, 0.172523, 0.291105, 0.181694,
  0.304399, 0.199489, 0.305238, 0.215147, 0.295207, 0.258848,
  0.290276, 0.280338, 0.284686, 0.304826, 0.282786, 0.313272,
  0.280879, 0.321717, 0.278333, 0.328622, 0.277125, 0.328622,
  0.275917, 0.328622, 0.263807, 0.298083, 0.250217, 0.260762,
  0.603168, 0.267455,
  0.596275, 0.224441, 0.596214, 0.225379, 0.607377, 0.193534,
  0.616190, 0.168443, 0.617319, 0.166991, 0.622168, 0.174996,
  0.627414, 0.183676, 0.634428, 0.218541, 0.634428, 0.235834,
  0.634428, 0.247209, 0.615733, 0.299998, 0.611555, 0.300400,
  0.609877, 0.300567, 0.606096, 0.285736, 0.603168, 0.267455,
  0.694143, 0.241583,
  0.687769, 0.192149, 0.689507, 0.184608, 0.705061, 0.194058,
  0.720035, 0.203146, 0.735071, 0.221259, 0.735071, 0.230185,
  0.735071, 0.236459, 0.704695, 0.283967, 0.700700, 0.283967,
  0.700090, 0.283967, 0.697132, 0.264893, 0.694143, 0.241583,
  0.330182, 0.271407,
  0.328505, 0.267567, 0.324357, 0.254254, 0.320911, 0.241812,
  0.314689, 0.219199,
  0.323595, 0.190955,
  0.332134, 0.163960, 0.333201, 0.162559, 0.348206, 0.159311,
  0.374038, 0.153706, 0.374526, 0.158356, 0.355343, 0.228253,
  0.342046, 0.276760, 0.336617, 0.286065, 0.330182, 0.271407,
  0.493315, 0.224167,
  0.482275, 0.191513, 0.475321, 0.161610, 0.477395, 0.155515,
  0.478401, 0.152484, 0.490845, 0.150000, 0.504996, 0.150000,
  0.528936, 0.150000, 0.530736, 0.150804, 0.530736, 0.161471,
  0.530736, 0.172506, 0.509509, 0.235086, 0.503410, 0.241962,
  0.501793, 0.243799, 0.497249, 0.235789, 0.493315, 0.224167,
  0.641046, 0.216447,
  0.632720, 0.175828, 0.634611, 0.168900, 0.651720, 0.177268,
  0.663462, 0.183012, 0.663858, 0.194695, 0.652940, 0.213841,
  0.643791, 0.229866,
  0.641046, 0.216447,
  0.413594, 0.206807,
  0.411154, 0.198139, 0.407006, 0.187790, 0.404383, 0.183799,
  0.401761, 0.179808, 0.399595, 0.172116, 0.399595, 0.166712,
  0.399595, 0.152836, 0.437687, 0.144731, 0.444457, 0.157162,
  0.448392, 0.164368, 0.447721, 0.168794, 0.438754, 0.193857,
  0.427226, 0.226138, 0.420090, 0.229805, 0.413594, 0.206807,
  0.581514, 0.187952,
  0.578434, 0.173606, 0.577580, 0.162492, 0.579319, 0.159339,
  0.583741, 0.151245, 0.603930, 0.160254, 0.603930, 0.170324,
  0.603930, 0.180187, 0.592524, 0.211401, 0.588925, 0.211401,
  0.587645, 0.211401, 0.584290, 0.200846, 0.581514, 0.187952,
  0.302740, 0.189152,
  0.296897, 0.172021, 0.298160, 0.166746, 0.308102, 0.166746,
  0.314872, 0.166746, 0.317251, 0.169353, 0.317251, 0.176793,
  0.317251, 0.186327, 0.311914, 0.200237, 0.308224, 0.200237,
  0.307278, 0.200237, 0.304820, 0.195242, 0.302740, 0.189152
};

// Extracted from: https://llvm.org/Logo.html (old logo).
constexpr char dragon_commands[] =
  "MLCCCCLCCCLCCCCCCCCCLCLCCCCCCCCLCCCLCCCCCCCCLLCCCLCCCCCCCCCCCCLCCCCCCCCCCCCCCCCCCCCCCCLLCLLCCCLLCCCCCLCLCCCLCCCCCCCCCLCCCCCCCLCCCCCLCCCCCCCCCCCCCCLLLCCCCCCLCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCLLCCLCLCCCCCCLLLCCCLCLLCLLCCCCCCLCLLLCCCCCCLLCCCCLLCCCLCCCCCCLLLCCCCCCLCLCCCCCCCCLLCLLCCLCCCCCCCCCLCCCCLCCCCCLCCZMCLZMCLCCLZMCLLCLCCLCCCZ";

constexpr double dragon_vertices[] = {
  0.867886, 0.000277,
  0.864345, 0.005538,
  0.846621, 0.031677, 0.830066, 0.059176, 0.814765, 0.087893,
  0.807847, 0.100544, 0.801637, 0.113854, 0.796184, 0.127714,
  0.791803, 0.138182, 0.788323, 0.149273, 0.785810, 0.160777,
  0.784906, 0.163906, 0.784440, 0.167228, 0.784435, 0.170580,
  0.784435, 0.170802,
  0.779145, 0.172954, 0.775573, 0.179511, 0.775685, 0.186863,
  0.775581, 0.193745, 0.778679, 0.200000, 0.783476, 0.202591,
  0.785060, 0.223139, 0.786809, 0.265839, 0.786393, 0.324434,
  0.786226, 0.332299,
  0.785039, 0.376212, 0.778945, 0.419646, 0.768144, 0.461175,
  0.767478, 0.460622, 0.766894, 0.460400, 0.766478, 0.460455,
  0.765228, 0.460622, 0.761853, 0.467101, 0.758562, 0.474357,
  0.756145, 0.479341, 0.753604, 0.485156, 0.752146, 0.490473,
  0.751353, 0.492925, 0.750929, 0.495559, 0.750896, 0.498227,
  0.750814, 0.499871, 0.750956, 0.501522, 0.751312, 0.503100,
  0.751729, 0.505150, 0.752229, 0.506313, 0.752229, 0.506313,
  0.752187, 0.506442, 0.752146, 0.506571, 0.752104, 0.506700,
  0.751483, 0.507784, 0.751081, 0.509062, 0.750937, 0.510411,
  0.750437, 0.512017,
  0.747170, 0.521946, 0.743584, 0.531684, 0.739688, 0.541204,
  0.736480, 0.549068,
  0.735524, 0.551204, 0.734552, 0.553327, 0.733564, 0.555437,
  0.733275, 0.555146, 0.732968, 0.554886, 0.732647, 0.554662,
  0.731008, 0.554310, 0.729329, 0.554784, 0.727939, 0.555991,
  0.725981, 0.557930, 0.721648, 0.566736, 0.707191, 0.576372,
  0.702275, 0.579640, 0.697775, 0.581523, 0.693859, 0.582686,
  0.699426, 0.544863, 0.702880, 0.506544, 0.704191, 0.468043,
  0.704842, 0.448459, 0.703948, 0.428839, 0.701525, 0.409503,
  0.701650, 0.408949, 0.701650, 0.408561, 0.701650, 0.408561,
  0.701442, 0.409115,
  0.700342, 0.400562, 0.698896, 0.392095, 0.697109, 0.383750,
  0.691460, 0.355523, 0.681156, 0.329321, 0.666944, 0.307044,
  0.640655, 0.265285, 0.604616, 0.243907, 0.572952, 0.242523,
  0.567744, 0.242412,
  0.567744, 0.241969, 0.567744, 0.241581, 0.567828, 0.241194,
  0.568703, 0.235766, 0.568828, 0.234160, 0.564203, 0.239421,
  0.563378, 0.240370, 0.562586, 0.241368, 0.561828, 0.242412,
  0.559620, 0.242467, 0.557620, 0.242633, 0.555745, 0.242800,
  0.555590, 0.240813, 0.555549, 0.238813, 0.555620, 0.236818,
  0.555787, 0.232609, 0.556620, 0.229785, 0.551996, 0.235046,
  0.549579, 0.237573, 0.547568, 0.240718, 0.546079, 0.244295,
  0.543996, 0.244738, 0.542746, 0.245126, 0.542746, 0.245126,
  0.544621, 0.245015,
  0.542663, 0.245458,
  0.541997, 0.243132, 0.540455, 0.239643, 0.540663, 0.236597,
  0.541080, 0.230062, 0.542455, 0.228843, 0.539122, 0.232997,
  0.535039, 0.237981, 0.530706, 0.245513, 0.529956, 0.249556,
  0.525956, 0.251218,
  0.525206, 0.250221, 0.522873, 0.247341, 0.522373, 0.241360,
  0.521832, 0.235268, 0.521873, 0.232166, 0.519040, 0.238535,
  0.516450, 0.244540, 0.514620, 0.251070, 0.513624, 0.257864,
  0.512291, 0.258695, 0.510999, 0.259525, 0.509749, 0.260411,
  0.508147, 0.257680, 0.507307, 0.254293, 0.507374, 0.250830,
  0.507166, 0.242855, 0.509041, 0.236375, 0.504583, 0.247064,
  0.501817, 0.254169, 0.499771, 0.261730, 0.498500, 0.269550,
  0.497000, 0.270934, 0.495542, 0.272374, 0.494125, 0.273925,
  0.493940, 0.270792, 0.493829, 0.267652, 0.493792, 0.264510,
  0.493709, 0.256645, 0.494667, 0.248172, 0.491001, 0.263125,
  0.489918, 0.267501, 0.488793, 0.273759, 0.488084, 0.280903,
  0.471252, 0.302226, 0.461003, 0.331690, 0.461253, 0.367966,
  0.461253, 0.368353,
  0.461212, 0.374833, 0.461462, 0.380981, 0.461837, 0.386630,
  0.462670, 0.399811, 0.464211, 0.410887, 0.466294, 0.420136,
  0.462336, 0.427336, 0.462003, 0.458295, 0.465878, 0.465440,
  0.468919, 0.471200, 0.486543, 0.484824, 0.490292, 0.487538,
  0.491822, 0.488925, 0.493149, 0.490670, 0.494209, 0.492688,
  0.499000, 0.503100, 0.499000, 0.524091, 0.495334, 0.534890,
  0.490876, 0.548071, 0.499792, 0.550674, 0.507208, 0.552613,
  0.514624, 0.554607, 0.524498, 0.551339, 0.526498, 0.546078,
  0.528456, 0.540816, 0.520082, 0.528964, 0.516624, 0.504651,
  0.513124, 0.480338, 0.522040, 0.478399, 0.522040, 0.478399,
  0.522040, 0.478399, 0.525998, 0.479729, 0.531914, 0.492190,
  0.536497, 0.501882, 0.541372, 0.509525, 0.545913, 0.515063,
  0.543583, 0.518893, 0.540745, 0.522121, 0.537539, 0.524589,
  0.542538, 0.524866, 0.547538, 0.521764, 0.550204, 0.519771,
  0.551155, 0.520672, 0.552143, 0.521504, 0.553162, 0.522263,
  0.552492, 0.523218, 0.551797, 0.524141, 0.551079, 0.525032,
  0.552608, 0.524845, 0.554117, 0.524436, 0.555579, 0.523814,
  0.556694, 0.524447, 0.557873, 0.524858, 0.559078, 0.525032,
  0.565536, 0.525697, 0.582826, 0.504651, 0.582826, 0.500719,
  0.582826, 0.499722, 0.582201, 0.498504, 0.581243, 0.497230,
  0.579493, 0.494738, 0.576244, 0.491691, 0.572619, 0.487427,
  0.570968, 0.485447, 0.569436, 0.483299, 0.568036, 0.481002,
  0.564653, 0.474969, 0.562617, 0.467790, 0.562161, 0.460289,
  0.562161, 0.459735,
  0.562036, 0.458018,
  0.562036, 0.458018, 0.562328, 0.457631, 0.562703, 0.456911,
  0.562703, 0.457132,
  0.562870, 0.456579,
  0.565036, 0.452148, 0.569119, 0.435588, 0.543288, 0.396876,
  0.532464, 0.380676, 0.519394, 0.367400, 0.504791, 0.357775,
  0.503041, 0.331634, 0.511041, 0.307210, 0.530872, 0.295192,
  0.530664, 0.295469,
  0.530997, 0.295137,
  0.540455, 0.289488, 0.552537, 0.286552, 0.567536, 0.287605,
  0.596284, 0.289598, 0.618365, 0.307487, 0.633781, 0.334403,
  0.655195, 0.373836, 0.664528, 0.432985, 0.657737, 0.492578,
  0.654904, 0.496177, 0.655445, 0.496731, 0.657195, 0.496842,
  0.656546, 0.501911, 0.655782, 0.506954, 0.654904, 0.511962,
  0.653779, 0.512792,
  0.648654, 0.516891, 0.650821, 0.516835, 0.653779, 0.517888,
  0.653529, 0.518995,
  0.652696, 0.522595, 0.651737, 0.526195, 0.650779, 0.529684,
  0.649552, 0.530160, 0.648342, 0.530715, 0.647154, 0.531346,
  0.641530, 0.534281, 0.643821, 0.534669, 0.646654, 0.536386,
  0.648488, 0.537604,
  0.647154, 0.542035, 0.645738, 0.546465, 0.644280, 0.550730,
  0.643157, 0.550962, 0.642044, 0.551277, 0.640947, 0.551671,
  0.635030, 0.553665, 0.637280, 0.554441, 0.639947, 0.556656,
  0.640519, 0.557120, 0.641075, 0.557619, 0.641613, 0.558151,
  0.640180, 0.562136, 0.638694, 0.566088, 0.637155, 0.570003,
  0.635000, 0.570175, 0.632865, 0.570639, 0.630781, 0.571388,
  0.624948, 0.573382, 0.627156, 0.574157, 0.629823, 0.576372,
  0.631239, 0.577480, 0.632322, 0.578643, 0.633197, 0.579640,
  0.631905, 0.582707, 0.630586, 0.585753, 0.629239, 0.588778,
  0.628281, 0.590772,
  0.625384, 0.590538, 0.622478, 0.590988, 0.619698, 0.592101,
  0.613782, 0.594095, 0.616032, 0.594870, 0.618698, 0.597086,
  0.620247, 0.598353, 0.621698, 0.599818, 0.623031, 0.601461,
  0.621314, 0.604813, 0.619564, 0.608137, 0.617782, 0.611430,
  0.614644, 0.610376, 0.611368, 0.610282, 0.608199, 0.611153,
  0.602200, 0.612261, 0.604366, 0.613368, 0.606824, 0.615916,
  0.609658, 0.618851, 0.611032, 0.621399, 0.611741, 0.622451,
  0.609158, 0.627103,
  0.607324, 0.623891, 0.602950, 0.620236, 0.598908, 0.616414,
  0.595257, 0.613026, 0.591158, 0.610583, 0.586826, 0.609214,
  0.584826, 0.609214, 0.578535, 0.624057, 0.544455, 0.620125,
  0.510332, 0.616193, 0.508208, 0.611817, 0.505208, 0.611817,
  0.502291, 0.611817, 0.495917, 0.614974, 0.495709, 0.615085,
  0.494292, 0.613977,
  0.480301, 0.602506, 0.466782, 0.590046, 0.453796, 0.576649,
  0.434464, 0.556157, 0.423798, 0.540816, 0.423798, 0.540816,
  0.423798, 0.540816, 0.429506, 0.533893, 0.421840, 0.525087,
  0.417404, 0.519568, 0.412700, 0.514443, 0.407758, 0.509746,
  0.404558, 0.506381, 0.401189, 0.503309, 0.397675, 0.500553,
  0.396415, 0.498244, 0.395193, 0.495899, 0.394009, 0.493519,
  0.388693, 0.482140, 0.383650, 0.470541, 0.378885, 0.458738,
  0.375210, 0.449140, 0.371736, 0.439407, 0.368469, 0.429552,
  0.361803, 0.409614, 0.357220, 0.391780, 0.354054, 0.376162,
  0.350934, 0.359380, 0.348872, 0.342281, 0.347888, 0.325044,
  0.347763, 0.320575, 0.347818, 0.316101, 0.348054, 0.311641,
  0.352199, 0.307338, 0.354626, 0.300742, 0.354637, 0.293752,
  0.354625, 0.282327, 0.348244, 0.272684, 0.339722, 0.271211,
  0.331222, 0.247563, 0.298808, 0.189410, 0.177985, 0.067401,
  0.172694, 0.061974,
  0.172902, 0.062195,
  0.171402, 0.060700,
  0.294475, 0.200819, 0.318473, 0.255593, 0.324806, 0.278466,
  0.324408, 0.279049, 0.324032, 0.279659, 0.323681, 0.280294,
  0.323431, 0.280054, 0.323182, 0.279814, 0.322931, 0.279574,
  0.323681, 0.279574, 0.323973, 0.279297, 0.323973, 0.279297,
  0.323973, 0.279297, 0.295975, 0.249058, 0.249521, 0.211176,
  0.202900, 0.173128, 0.176652, 0.164156, 0.171194, 0.162605,
  0.170069, 0.162273,
  0.170069, 0.162273, 0.178652, 0.172795, 0.177319, 0.178057,
  0.176902, 0.179829, 0.170319, 0.179275, 0.159778, 0.179607,
  0.149641, 0.179660, 0.139530, 0.180958, 0.129573, 0.183484,
  0.106225, 0.188966, 0.083991, 0.200848, 0.064286, 0.218376,
  0.009916, 0.266504, 0.000000, 0.300675, 0.000000, 0.300675,
  0.000000, 0.300675, 0.022915, 0.258141, 0.065828, 0.228400,
  0.065245, 0.247064, 0.063870, 0.337449, 0.084368, 0.456579,
  0.106491, 0.585067, 0.148154, 0.705083, 0.151612, 0.714941,
  0.151779, 0.724799, 0.152279, 0.730005, 0.152279, 0.730005,
  0.152279, 0.730005, 0.158445, 0.630869, 0.183776, 0.528023,
  0.191067, 0.543696, 0.215357, 0.589886, 0.270269, 0.644106,
  0.338097, 0.711119, 0.369011, 0.729174, 0.370261, 0.729839,
  0.370302, 0.738534, 0.370469, 0.743131, 0.370469, 0.743131,
  0.370469, 0.743131, 0.370677, 0.737593, 0.371219, 0.728565,
  0.373052, 0.724965, 0.383635, 0.704750, 0.394217, 0.691458,
  0.401112, 0.682479, 0.408647, 0.674411, 0.416715, 0.667367,
  0.418048, 0.673459, 0.419340, 0.678665, 0.420590, 0.682652,
  0.419757, 0.677280, 0.419090, 0.672130, 0.418423, 0.667034,
  0.422881, 0.667588, 0.440838, 0.670690, 0.468669, 0.688800,
  0.482960, 0.698104, 0.494042, 0.710676, 0.501917, 0.721753,
  0.501718, 0.721852, 0.501524, 0.721963, 0.501333, 0.722085,
  0.501143, 0.722208, 0.506458, 0.728288, 0.512541, 0.744405,
  0.502283, 0.743885, 0.492027, 0.745454, 0.482126, 0.749057,
  0.480862, 0.745503, 0.479500, 0.742011, 0.478043, 0.738589,
  0.477778, 0.736005, 0.477231, 0.733487, 0.476419, 0.731113,
  0.475294, 0.727734, 0.472461, 0.725574, 0.469586, 0.723359,
  0.468314, 0.721711, 0.466851, 0.720346, 0.465253, 0.719316,
  0.464289, 0.718191, 0.463509, 0.716815, 0.462961, 0.715273,
  0.459462, 0.705083, 0.429714, 0.715938, 0.423715, 0.725242,
  0.416424, 0.736540, 0.415007, 0.750940, 0.417882, 0.758915,
  0.418757, 0.761407, 0.420798, 0.763456, 0.423631, 0.765007,
  0.423715, 0.777247, 0.429339, 0.806545, 0.440463, 0.827812,
  0.438334, 0.815648, 0.436831, 0.803306, 0.435964, 0.790871,
  0.435589, 0.782010, 0.435964, 0.775696, 0.436589, 0.771487,
  0.438547, 0.789653, 0.443838, 0.809369, 0.460628, 0.833793,
  0.450296, 0.803886, 0.451462, 0.786053, 0.452504, 0.770435,
  0.452879, 0.767998,
  0.453129, 0.767942,
  0.454629, 0.782508, 0.459045, 0.797517, 0.473211, 0.821332,
  0.473001, 0.820825, 0.472807, 0.820308, 0.472627, 0.819781,
  0.473211, 0.820944,
  0.466240, 0.803462, 0.463795, 0.783414, 0.466253, 0.763900,
  0.466586, 0.763733,
  0.468503, 0.765395, 0.470377, 0.767610, 0.472169, 0.770712,
  0.477502, 0.780238, 0.480002, 0.790040, 0.481002, 0.796742,
  0.479857, 0.804045, 0.480856, 0.811653, 0.483793, 0.818009,
  0.473586, 0.827922, 0.463586, 0.837171, 0.453796, 0.845756,
  0.444213, 0.854008, 0.433381, 0.863091, 0.422256, 0.871786,
  0.416424, 0.876217, 0.410632, 0.880536, 0.405008, 0.884524,
  0.399383, 0.880259,
  0.399467, 0.880259,
  0.399133, 0.880038,
  0.358720, 0.849079, 0.328723, 0.828421, 0.294225, 0.820113,
  0.255437, 0.810864, 0.219773, 0.835787, 0.211607, 0.880038,
  0.206275, 0.909422, 0.215499, 0.940077, 0.234605, 0.956467,
  0.235105, 0.956910,
  0.235647, 0.957464, 0.236189, 0.957851, 0.236772, 0.958350,
  0.239272, 0.960343,
  0.239355, 0.960343,
  0.247354, 0.966436, 0.257020, 0.970922, 0.268144, 0.973469,
  0.269311, 0.973746,
  0.269561, 0.973746,
  0.275241, 0.975008, 0.280973, 0.975822, 0.286726, 0.976183,
  0.327348, 0.978786, 0.376552, 0.953587, 0.425173, 0.921797,
  0.429339, 0.924788, 0.433589, 0.927778, 0.437922, 0.930990,
  0.544038, 1.008693, 0.608658, 1.018773, 0.694317, 0.971309,
  0.673402, 0.979672, 0.643988, 0.991026, 0.607074, 0.990361,
  0.606817, 0.990174, 0.606533, 0.990060, 0.606241, 0.990029,
  0.606283, 0.990361,
  0.597250, 0.990151, 0.588235, 0.989227, 0.579285, 0.987592,
  0.579285, 0.987537,
  0.578035, 0.987370,
  0.578035, 0.987204,
  0.578021, 0.987241, 0.578007, 0.987278, 0.577994, 0.987315,
  0.555689, 0.983029, 0.533898, 0.974912, 0.513166, 0.963168,
  0.502559, 0.957221, 0.492198, 0.950529, 0.482126, 0.943119,
  0.468076, 0.932803, 0.454104, 0.922298, 0.440213, 0.911606,
  0.454712, 0.901748, 0.469044, 0.891502, 0.482960, 0.881423,
  0.497750, 0.870955, 0.510957, 0.861374, 0.521207, 0.853731,
  0.521582, 0.853509,
  0.521873, 0.853454,
  0.521873, 0.853454, 0.525331, 0.852457, 0.528539, 0.851128,
  0.535622, 0.848248, 0.545621, 0.842876, 0.557495, 0.835455,
  0.558662, 0.841934, 0.561453, 0.847362, 0.565620, 0.849466,
  0.575410, 0.854506, 0.582576, 0.850962, 0.587368, 0.843817,
  0.588118, 0.843264,
  0.590076, 0.841214,
  0.592159, 0.839553, 0.594534, 0.837504, 0.596575, 0.836230,
  0.597304, 0.835799, 0.598040, 0.835393, 0.598783, 0.835011,
  0.600450, 0.834458, 0.602033, 0.833738, 0.603616, 0.832962,
  0.604991, 0.832408,
  0.598950, 0.848857, 0.601200, 0.864586, 0.601200, 0.864586,
  0.602991, 0.844039, 0.610366, 0.830913, 0.617532, 0.824987,
  0.620198, 0.823492, 0.622948, 0.822218, 0.625865, 0.821276,
  0.628531, 0.821276, 0.630697, 0.822550, 0.631864, 0.825153,
  0.634780, 0.831799, 0.640905, 0.835898, 0.647029, 0.837670,
  0.648029, 0.845977, 0.645155, 0.860045, 0.630364, 0.885632,
  0.634114, 0.882198,
  0.634280, 0.882032,
  0.637280, 0.879096,
  0.643238, 0.873115, 0.647779, 0.867466, 0.651196, 0.862205,
  0.654618, 0.857283, 0.657317, 0.851554, 0.659153, 0.845313,
  0.659653, 0.856389, 0.657362, 0.873392, 0.646196, 0.902690,
  0.646196, 0.902690, 0.659403, 0.887404, 0.667611, 0.867798,
  0.667861, 0.878376, 0.667194, 0.892333, 0.664986, 0.911274,
  0.679360, 0.874001, 0.678527, 0.851183, 0.674944, 0.837836,
  0.674527, 0.835842,
  0.674457, 0.834630, 0.674318, 0.833427, 0.674110, 0.832242,
  0.675860, 0.829085,
  0.680777, 0.819892, 0.673861, 0.810034, 0.669444, 0.806101,
  0.666944, 0.803886, 0.661112, 0.797572, 0.650529, 0.797406,
  0.641155, 0.782453, 0.629281, 0.789099, 0.610616, 0.805437,
  0.608408, 0.807377, 0.606097, 0.809098, 0.603700, 0.810588,
  0.603083, 0.808341, 0.602316, 0.806173, 0.601408, 0.804108,
  0.608074, 0.798791, 0.614990, 0.793086, 0.621990, 0.787050,
  0.635197, 0.775585, 0.645030, 0.764010, 0.652321, 0.752989,
  0.673569, 0.724356, 0.681068, 0.696498, 0.702733, 0.672905,
  0.702191, 0.672960,
  0.704691, 0.671188,
  0.740272, 0.645823, 0.759145, 0.640894, 0.760478, 0.640561,
  0.760353, 0.647650,
  0.761103, 0.640727,
  0.768530, 0.645562, 0.775302, 0.652000, 0.781143, 0.659779,
  0.787118, 0.667762, 0.792197, 0.676845, 0.796225, 0.686751,
  0.798058, 0.703421,
  0.798058, 0.703421, 0.798058, 0.696941, 0.797642, 0.685145,
  0.804016, 0.676726, 0.840138, 0.627325, 0.869469, 0.548515,
  0.898383, 0.470757, 0.908799, 0.408783, 0.911841, 0.386574,
  0.923472, 0.431180, 0.932730, 0.476817, 0.939547, 0.523149,
  0.946363, 0.569481, 0.939463, 0.515783, 0.938255, 0.502436,
  0.942296, 0.483827, 0.959753, 0.398759, 0.964378, 0.294417,
  0.968544, 0.201318, 0.961670, 0.140452, 0.958545, 0.118631,
  0.985543, 0.148371, 1.000000, 0.191016, 1.000000, 0.191016,
  1.000000, 0.191016, 0.998333, 0.181989, 0.992167, 0.166814,
  0.993001, 0.168309,
  0.993001, 0.168309, 0.983168, 0.143110, 0.966169, 0.118188,
  0.964032, 0.114960, 0.961852, 0.111784, 0.959628, 0.108662,
  0.941630, 0.083241, 0.924048, 0.074490, 0.910882, 0.071444,
  0.906716, 0.070337, 0.902425, 0.069783, 0.898050, 0.069783,
  0.897259, 0.069783,
  0.892051, 0.069617, 0.888884, 0.069894, 0.888634, 0.068343,
  0.887801, 0.063081, 0.893217, 0.052559, 0.893217, 0.052559,
  0.893217, 0.052559, 0.885843, 0.055106, 0.870136, 0.070835,
  0.860553, 0.080084, 0.848471, 0.093542, 0.834597, 0.112926,
  0.821570, 0.130482, 0.809195, 0.148873, 0.797517, 0.168032,
  0.794392, 0.173294,
  0.794090, 0.172950, 0.793770, 0.172636, 0.793434, 0.172352,
  0.796600, 0.154020, 0.808516, 0.110434, 0.867928, 0.000000,
  0.496667, 0.615805,
  0.496835, 0.615912, 0.497002, 0.616022, 0.497167, 0.616137,
  0.496667, 0.615805,
  0.485710, 0.760133,
  0.489376, 0.762570, 0.493626, 0.766004, 0.498333, 0.770878,
  0.498125, 0.770878,
  0.495817, 0.771653, 0.493631, 0.772966, 0.491667, 0.774755,
  0.489505, 0.770952, 0.487740, 0.766779, 0.486418, 0.762349,
  0.485710, 0.760133,
  0.283435, 0.847085,
  0.288934, 0.847085, 0.294934, 0.847860, 0.301433, 0.849743,
  0.302266, 0.850020,
  0.302975, 0.850242,
  0.307558, 0.851737, 0.312266, 0.853676, 0.317015, 0.855891,
  0.319390, 0.857054,
  0.327348, 0.861152, 0.335847, 0.865694, 0.344846, 0.870789,
  0.358178, 0.878487, 0.370636, 0.886130, 0.379760, 0.891558,
  0.387384, 0.896431,
  0.378177, 0.902302, 0.369719, 0.907286, 0.363345, 0.910886,
  0.321098, 0.933926, 0.287101, 0.940572, 0.265936, 0.932541,
  0.226564, 0.917643, 0.234272, 0.847473, 0.283435, 0.847140
};

// Public domain.
constexpr char world_commands[] =
  "MLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLLLLLLLLZMLLLLLLLLLLLLLLZ";

constexpr double world_vertices[] = {
  0.442757, 0.379556,
  0.438444, 0.387420,
  0.435571, 0.391353,
  0.429823, 0.395678,
  0.428960, 0.399218,
  0.426949, 0.404332,
  0.427238, 0.409837,
  0.421488, 0.416522,
  0.417463, 0.421635,
  0.415165, 0.424782,
  0.411716, 0.430287,
  0.408841, 0.435398,
  0.405679, 0.443658,
  0.402517, 0.453096,
  0.401941, 0.455062,
  0.402805, 0.465289,
  0.403379, 0.471974,
  0.402230, 0.481413,
  0.399357, 0.490065,
  0.400506, 0.495569,
  0.400793, 0.501863,
  0.401081, 0.505402,
  0.405680, 0.509729,
  0.409130, 0.514841,
  0.412004, 0.522707,
  0.415166, 0.529392,
  0.418615, 0.534900,
  0.423215, 0.540404,
  0.427525, 0.543942,
  0.431549, 0.545909,
  0.436148, 0.545909,
  0.441323, 0.544337,
  0.445634, 0.541976,
  0.449083, 0.542369,
  0.453106, 0.540799,
  0.457992, 0.539224,
  0.462880, 0.536862,
  0.466902, 0.536077,
  0.470352, 0.535684,
  0.474663, 0.536862,
  0.477539, 0.538438,
  0.480124, 0.542763,
  0.482136, 0.545911,
  0.483861, 0.546303,
  0.486736, 0.545518,
  0.489897, 0.546699,
  0.492483, 0.550630,
  0.493347, 0.552203,
  0.494783, 0.560855,
  0.493634, 0.569506,
  0.492197, 0.575800,
  0.493634, 0.580519,
  0.498808, 0.587989,
  0.502256, 0.596643,
  0.504270, 0.604507,
  0.507143, 0.612375,
  0.507718, 0.619059,
  0.507431, 0.624171,
  0.509156, 0.630858,
  0.508293, 0.637150,
  0.507718, 0.639510,
  0.505132, 0.646199,
  0.503407, 0.649735,
  0.501970, 0.659174,
  0.502832, 0.667037,
  0.507718, 0.678838,
  0.510304, 0.688275,
  0.512316, 0.694569,
  0.513178, 0.701254,
  0.513467, 0.709513,
  0.514616, 0.715807,
  0.515478, 0.720528,
  0.519502, 0.726820,
  0.521227, 0.730753,
  0.523527, 0.737438,
  0.524964, 0.748449,
  0.526975, 0.753956,
  0.531288, 0.756315,
  0.536462, 0.753563,
  0.541060, 0.753563,
  0.545373, 0.751203,
  0.549973, 0.748844,
  0.553995, 0.746089,
  0.558881, 0.739405,
  0.562905, 0.734291,
  0.568367, 0.727211,
  0.571527, 0.721707,
  0.573252, 0.716986,
  0.574978, 0.712268,
  0.579579, 0.704009,
  0.582739, 0.700076,
  0.584464, 0.697715,
  0.583891, 0.687885,
  0.582740, 0.684735,
  0.586190, 0.672937,
  0.591939, 0.664287,
  0.595676, 0.660354,
  0.601999, 0.652882,
  0.603724, 0.644621,
  0.603151, 0.635576,
  0.602000, 0.624171,
  0.598838, 0.608047,
  0.598262, 0.599395,
  0.601138, 0.592317,
  0.605737, 0.584449,
  0.613497, 0.571080,
  0.618957, 0.561249,
  0.625856, 0.552201,
  0.632181, 0.539616,
  0.634765, 0.534505,
  0.638503, 0.526639,
  0.641375, 0.509334,
  0.636202, 0.511300,
  0.629591, 0.512481,
  0.616946, 0.513660,
  0.611483, 0.511300,
  0.608323, 0.498716,
  0.602862, 0.488096,
  0.598261, 0.481413,
  0.593950, 0.473546,
  0.589639, 0.463321,
  0.585902, 0.451522,
  0.583315, 0.442870,
  0.575267, 0.420059,
  0.573255, 0.413767,
  0.577567, 0.418880,
  0.580729, 0.417698,
  0.585040, 0.423597,
  0.592515, 0.438150,
  0.595964, 0.449161,
  0.600276, 0.459388,
  0.605735, 0.468827,
  0.611773, 0.484164,
  0.614646, 0.497535,
  0.620107, 0.501076,
  0.627006, 0.498716,
  0.638503, 0.489668,
  0.648852, 0.483770,
  0.655173, 0.476300,
  0.661211, 0.466073,
  0.665233, 0.459780,
  0.668684, 0.450341,
  0.662648, 0.444835,
  0.658048, 0.436576,
  0.654023, 0.431465,
  0.647124, 0.440119,
  0.639077, 0.441297,
  0.633902, 0.433824,
  0.631604, 0.425173,
  0.629304, 0.416912,
  0.630166, 0.407080,
  0.637640, 0.416519,
  0.646550, 0.423206,
  0.653448, 0.423990,
  0.656035, 0.424383,
  0.658621, 0.426349,
  0.663794, 0.431855,
  0.668970, 0.433820,
  0.677882, 0.434608,
  0.683341, 0.434214,
  0.689665, 0.434998,
  0.695414, 0.438538,
  0.699725, 0.445225,
  0.704898, 0.450730,
  0.708635, 0.457416,
  0.715248, 0.455846,
  0.718123, 0.455846,
  0.720133, 0.465285,
  0.720133, 0.475903,
  0.728182, 0.501466,
  0.730481, 0.508941,
  0.732492, 0.517985,
  0.734792, 0.524278,
  0.739104, 0.524278,
  0.743990, 0.518771,
  0.745428, 0.514051,
  0.745428, 0.505008,
  0.744566, 0.493212,
  0.748302, 0.481019,
  0.752325, 0.473153,
  0.761237, 0.465680,
  0.767273, 0.456636,
  0.774459, 0.450737,
  0.776472, 0.451524,
  0.782507, 0.453491,
  0.789406, 0.464109,
  0.792856, 0.471976,
  0.798892, 0.480233,
  0.800330, 0.488492,
  0.805214, 0.484954,
  0.808662, 0.490460,
  0.811251, 0.503045,
  0.810962, 0.508549,
  0.811251, 0.518774,
  0.813549, 0.525856,
  0.814697, 0.514450,
  0.816135, 0.501081,
  0.820159, 0.502261,
  0.824470, 0.509339,
  0.829645, 0.516810,
  0.833381, 0.522316,
  0.839418, 0.519564,
  0.848330, 0.505011,
  0.846603, 0.493215,
  0.837118, 0.475517,
  0.833097, 0.462539,
  0.840857, 0.452708,
  0.847178, 0.462147,
  0.850053, 0.469225,
  0.854938, 0.455857,
  0.857526, 0.452710,
  0.862701, 0.447598,
  0.870749, 0.444058,
  0.878222, 0.431472,
  0.880809, 0.418888,
  0.876784, 0.403156,
  0.872763, 0.398045,
  0.865863, 0.386246,
  0.865574, 0.380347,
  0.868450, 0.374841,
  0.866724, 0.371696,
  0.861838, 0.372091,
  0.853217, 0.366586,
  0.860400, 0.353216,
  0.865862, 0.352821,
  0.868162, 0.361471,
  0.874198, 0.357147,
  0.877646, 0.359899,
  0.879084, 0.368158,
  0.881096, 0.369338,
  0.887706, 0.377597,
  0.889719, 0.386249,
  0.895469, 0.385070,
  0.900066, 0.376024,
  0.895755, 0.369338,
  0.886559, 0.360687,
  0.887132, 0.350460,
  0.889144, 0.343775,
  0.894318, 0.338661,
  0.899492, 0.338269,
  0.900356, 0.336695,
  0.904377, 0.323323,
  0.903803, 0.316245,
  0.903517, 0.302087,
  0.897480, 0.292648,
  0.888282, 0.285961,
  0.875635, 0.285177,
  0.871899, 0.280456,
  0.873625, 0.264723,
  0.879371, 0.258038,
  0.888859, 0.256073,
  0.902656, 0.258828,
  0.909267, 0.258828,
  0.910991, 0.248601,
  0.916164, 0.244273,
  0.922779, 0.246239,
  0.925940, 0.242308,
  0.927086, 0.243882,
  0.929674, 0.245451,
  0.930824, 0.255678,
  0.927376, 0.264723,
  0.930536, 0.275738,
  0.936285, 0.282423,
  0.940597, 0.291076,
  0.945483, 0.297368,
  0.948933, 0.298156,
  0.948933, 0.288717,
  0.949505, 0.276524,
  0.945770, 0.266693,
  0.938873, 0.261187,
  0.946059, 0.251355,
  0.950942, 0.248996,
  0.954968, 0.251355,
  0.959567, 0.244277,
  0.964740, 0.244277,
  0.967039, 0.241132,
  0.957844, 0.234838,
  0.951805, 0.230121,
  0.955256, 0.228548,
  0.958420, 0.224220,
  0.971642, 0.228548,
  0.979975, 0.229727,
  0.974227, 0.222647,
  0.960719, 0.218321,
  0.952096, 0.218713,
  0.943760, 0.214386,
  0.915306, 0.208095,
  0.911568, 0.214780,
  0.905243, 0.212026,
  0.888572, 0.208488,
  0.874490, 0.206127,
  0.863281, 0.203767,
  0.850059, 0.203767,
  0.840000, 0.202980,
  0.835114, 0.199442,
  0.830227, 0.197476,
  0.825916, 0.195902,
  0.823616, 0.200621,
  0.819016, 0.205341,
  0.812405, 0.204948,
  0.807231, 0.203375,
  0.800620, 0.202980,
  0.790560, 0.200228,
  0.784236, 0.196688,
  0.777338, 0.195902,
  0.765841, 0.199442,
  0.755494, 0.197868,
  0.745433, 0.195114,
  0.735085, 0.193148,
  0.739109, 0.186463,
  0.740545, 0.183318,
  0.734223, 0.180958,
  0.725887, 0.181744,
  0.718128, 0.179385,
  0.703757, 0.180958,
  0.698295, 0.182530,
  0.685648, 0.190395,
  0.674152, 0.192360,
  0.666390, 0.197866,
  0.668978, 0.207698,
  0.663230, 0.212024,
  0.656905, 0.201406,
  0.659779, 0.220283,
  0.654029, 0.218709,
  0.653742, 0.209270,
  0.651730, 0.202585,
  0.648569, 0.197866,
  0.644545, 0.195505,
  0.638510, 0.199439,
  0.642531, 0.208484,
  0.645981, 0.215169,
  0.632472, 0.216350,
  0.627296, 0.213597,
  0.620400, 0.210057,
  0.612926, 0.215955,
  0.605741, 0.216350,
  0.597406, 0.216350,
  0.587920, 0.221463,
  0.578434, 0.228541,
  0.566938, 0.236405,
  0.558314, 0.223427,
  0.566938, 0.222641,
  0.573547, 0.222641,
  0.573260, 0.216349,
  0.564063, 0.212809,
  0.556589, 0.209269,
  0.543941, 0.206124,
  0.536182, 0.205730,
  0.528708, 0.204944,
  0.521235, 0.208091,
  0.512899, 0.215955,
  0.502840, 0.226182,
  0.494504, 0.236800,
  0.492778, 0.242699,
  0.486455, 0.244665,
  0.484729, 0.245058,
  0.485017, 0.261182,
  0.491915, 0.258825,
  0.501975, 0.269441,
  0.506861, 0.275735,
  0.510310, 0.274160,
  0.516634, 0.265903,
  0.520658, 0.256857,
  0.517782, 0.250958,
  0.519220, 0.241911,
  0.524682, 0.234438,
  0.533879, 0.232081,
  0.533590, 0.236012,
  0.528991, 0.245451,
  0.533014, 0.252531,
  0.539052, 0.251350,
  0.544512, 0.250958,
  0.545949, 0.256071,
  0.539052, 0.259611,
  0.535314, 0.261183,
  0.533301, 0.270622,
  0.528416, 0.272196,
  0.523817, 0.282027,
  0.521231, 0.281633,
  0.517207, 0.281239,
  0.512033, 0.284779,
  0.508584, 0.285173,
  0.505709, 0.282027,
  0.499387, 0.282027,
  0.493925, 0.283206,
  0.492200, 0.280847,
  0.487314, 0.287141,
  0.486452, 0.287532,
  0.479841, 0.289893,
  0.477254, 0.296971,
  0.473230, 0.298544,
  0.468344, 0.305623,
  0.464895, 0.307196,
  0.461157, 0.308375,
  0.458284, 0.308769,
  0.453397, 0.311129,
  0.456559, 0.316635,
  0.452822, 0.319780,
  0.456271, 0.323320,
  0.458570, 0.328434,
  0.459433, 0.330793,
  0.458570, 0.337873,
  0.455984, 0.339838,
  0.449946, 0.341018,
  0.440749, 0.337086,
  0.437013, 0.338657,
  0.434140, 0.348096,
  0.433851, 0.358716,
  0.433562, 0.366975,
  0.435864, 0.372089,
  0.440749, 0.373269,
  0.445635, 0.374447,
  0.451672, 0.374841,
  0.460584, 0.367370,
  0.462595, 0.361864,
  0.465757, 0.352426,
  0.470355, 0.350065,
  0.473805, 0.342987,
  0.478116, 0.339054,
  0.487027, 0.338269,
  0.488463, 0.333548,
  0.491625, 0.334728,
  0.493636, 0.335123,
  0.497373, 0.342594,
  0.501685, 0.345739,
  0.506285, 0.348493,
  0.511170, 0.353998,
  0.513471, 0.357931,
  0.514907, 0.360292,
  0.510022, 0.367370,
  0.506285, 0.368156,
  0.510596, 0.370515,
  0.516919, 0.361470,
  0.516919, 0.359502,
  0.518357, 0.355571,
  0.521231, 0.356752,
  0.520943, 0.354786,
  0.514332, 0.346527,
  0.508294, 0.342199,
  0.505421, 0.337875,
  0.502257, 0.332760,
  0.503121, 0.327255,
  0.511168, 0.333153,
  0.516055, 0.338659,
  0.520941, 0.342987,
  0.525254, 0.348100,
  0.526402, 0.356359,
  0.529276, 0.360292,
  0.531862, 0.367370,
  0.534449, 0.372091,
  0.536461, 0.372484,
  0.538473, 0.371303,
  0.538759, 0.366977,
  0.541636, 0.364616,
  0.537324, 0.357538,
  0.538759, 0.353214,
  0.544797, 0.353998,
  0.550545, 0.353998,
  0.550545, 0.364616,
  0.553709, 0.370910,
  0.558594, 0.374843,
  0.564631, 0.375631,
  0.570380, 0.375238,
  0.577853, 0.375631,
  0.582452, 0.376811,
  0.583028, 0.382709,
  0.581015, 0.393328,
  0.575266, 0.403947,
  0.570382, 0.405127,
  0.560896, 0.403947,
  0.547387, 0.402766,
  0.538762, 0.399621,
  0.533014, 0.395688,
  0.531290, 0.396083,
  0.528128, 0.405915,
  0.525827, 0.407094,
  0.517205, 0.404342,
  0.508008, 0.397656,
  0.501109, 0.394904,
  0.495360, 0.388610,
  0.495648, 0.380745,
  0.497660, 0.373667,
  0.495360, 0.371702,
  0.488175, 0.374454,
  0.482138, 0.375635,
  0.467766, 0.378387,
  0.465466, 0.379173,
  0.460005, 0.381927,
  0.455120, 0.381533,
  0.450234, 0.381533,
  0.445059, 0.383497,
  0.442722, 0.379549,
  0.635048, 0.638326,
  0.631024, 0.648944,
  0.625274, 0.652877,
  0.622113, 0.655631,
  0.618089, 0.657991,
  0.616364, 0.660742,
  0.617226, 0.669787,
  0.616940, 0.676476,
  0.614640, 0.684338,
  0.612053, 0.688666,
  0.612341, 0.699678,
  0.616079, 0.707544,
  0.619528, 0.706364,
  0.624127, 0.703614,
  0.626425, 0.700074,
  0.627287, 0.689454,
  0.629876, 0.681983,
  0.632750, 0.673331,
  0.635049, 0.665861,
  0.636198, 0.656422,
  0.637062, 0.651703,
  0.637062, 0.646984,
  0.635018, 0.638317,
  0.860106, 0.691425,
  0.858095, 0.711875,
  0.856947, 0.726425,
  0.857809, 0.740191,
  0.856657, 0.751203,
  0.854648, 0.752775,
  0.856657, 0.756314,
  0.861546, 0.755920,
  0.868157, 0.753560,
  0.873904, 0.751199,
  0.881379, 0.750020,
  0.886264, 0.746482,
  0.894601, 0.743728,
  0.899775, 0.740190,
  0.908399, 0.739793,
  0.915008, 0.739793,
  0.920759, 0.747266,
  0.920468, 0.753560,
  0.923918, 0.753165,
  0.930242, 0.748446,
  0.933116, 0.746482,
  0.930816, 0.754739,
  0.931390, 0.759067,
  0.933116, 0.764571,
  0.933116, 0.770077,
  0.934263, 0.773222,
  0.939151, 0.777157,
  0.942888, 0.775584,
  0.948060, 0.777552,
  0.953523, 0.776372,
  0.967030, 0.765751,
  0.971342, 0.759459,
  0.975079, 0.754739,
  0.981692, 0.746087,
  0.987729, 0.735862,
  0.989740, 0.729566,
  0.992326, 0.718950,
  0.992900, 0.712265,
  0.992326, 0.701649,
  0.990601, 0.694567,
  0.985715, 0.691029,
  0.981977, 0.681197,
  0.977666, 0.672936,
  0.975079, 0.670185,
  0.975943, 0.662318,
  0.974794, 0.654848,
  0.972494, 0.646587,
  0.969619, 0.639902,
  0.967895, 0.634789,
  0.966458, 0.634394,
  0.964731, 0.646978,
  0.961569, 0.655632,
  0.958409, 0.662316,
  0.954959, 0.663888,
  0.951223, 0.663104,
  0.946910, 0.658383,
  0.944036, 0.654846,
  0.942025, 0.650518,
  0.942887, 0.643833,
  0.944899, 0.638719,
  0.943461, 0.636360,
  0.938574, 0.635965,
  0.933400, 0.635179,
  0.931389, 0.635572,
  0.924778, 0.642654,
  0.922769, 0.645406,
  0.920180, 0.647766,
  0.919890, 0.654058,
  0.912995, 0.651306,
  0.913281, 0.648946,
  0.909259, 0.645407,
  0.904947, 0.649341,
  0.902934, 0.654451,
  0.899485, 0.657598,
  0.898337, 0.663497,
  0.896326, 0.661533,
  0.895177, 0.659960,
  0.893165, 0.664678,
  0.891153, 0.670972,
  0.888564, 0.673725,
  0.881669, 0.678052,
  0.876208, 0.680804,
  0.871321, 0.682771,
  0.866721, 0.685916,
  0.862411, 0.689456,
  0.860164, 0.691396,
  0.877065, 0.536076,
  0.870744, 0.545909,
  0.866719, 0.547875,
  0.862120, 0.552988,
  0.858956, 0.560459,
  0.855508, 0.564787,
  0.851771, 0.569115,
  0.852921, 0.576586,
  0.855221, 0.583269,
  0.859244, 0.587599,
  0.867578, 0.587599,
  0.871316, 0.588779,
  0.873902, 0.587992,
  0.876202, 0.583666,
  0.876776, 0.579731,
  0.880801, 0.573832,
  0.881089, 0.569504,
  0.883389, 0.566753,
  0.882813, 0.562425,
  0.881375, 0.554165,
  0.883389, 0.547087,
  0.882813, 0.541187,
  0.878502, 0.536859,
  0.877084, 0.536055,
  0.933406, 0.575407,
  0.938869, 0.578160,
  0.941169, 0.587204,
  0.946052, 0.590353,
  0.950938, 0.584847,
  0.953238, 0.581307,
  0.956404, 0.582490,
  0.961576, 0.585240,
  0.970775, 0.589960,
  0.977385, 0.593893,
  0.981983, 0.597433,
  0.985145, 0.604904,
  0.987158, 0.608049,
  0.988596, 0.614732,
  0.997504, 0.624566,
  0.999232, 0.628894,
  0.991471, 0.627713,
  0.986583, 0.622994,
  0.980263, 0.615915,
  0.976813, 0.612769,
  0.971926, 0.617881,
  0.968189, 0.617488,
  0.961291, 0.615520,
  0.957844, 0.612769,
  0.954970, 0.610803,
  0.955256, 0.605295,
  0.954109, 0.600578,
  0.950084, 0.596250,
  0.944911, 0.593498,
  0.943760, 0.593105,
  0.940598, 0.591925,
  0.935425, 0.589958,
  0.934849, 0.585239,
  0.934849, 0.581306,
  0.930251, 0.576588,
  0.930537, 0.575407,
  0.933410, 0.575389,
  0.804059, 0.542750,
  0.807795, 0.544325,
  0.813257, 0.552975,
  0.818717, 0.559269,
  0.825903, 0.565168,
  0.830214, 0.573425,
  0.834525, 0.582476,
  0.836827, 0.587980,
  0.839124, 0.593879,
  0.837976, 0.600957,
  0.834813, 0.604104,
  0.830214, 0.599384,
  0.825615, 0.590341,
  0.820154, 0.578148,
  0.816418, 0.571068,
  0.812106, 0.560842,
  0.808656, 0.554943,
  0.806070, 0.548258,
  0.803197, 0.543142,
  0.804040, 0.542757,
  0.245013, 0.849518,
  0.237253, 0.843619,
  0.231505, 0.833391,
  0.228342, 0.825131,
  0.225469, 0.816481,
  0.228056, 0.811368,
  0.228342, 0.805863,
  0.226042, 0.800750,
  0.224031, 0.793277,
  0.220869, 0.786985,
  0.218569, 0.783838,
  0.216844, 0.751982,
  0.215695, 0.734284,
  0.215119, 0.718553,
  0.214544, 0.703216,
  0.214257, 0.687880,
  0.214257, 0.671363,
  0.210520, 0.665463,
  0.203047, 0.660745,
  0.194136, 0.650910,
  0.186663, 0.642259,
  0.182639, 0.632032,
  0.179190, 0.620629,
  0.174591, 0.610011,
  0.167692, 0.598607,
  0.168267, 0.592315,
  0.172003, 0.587596,
  0.170278, 0.582876,
  0.170278, 0.574224,
  0.174589, 0.565963,
  0.178900, 0.556133,
  0.182351, 0.547478,
  0.184362, 0.538435,
  0.182062, 0.530173,
  0.184362, 0.523882,
  0.191548, 0.512870,
  0.195859, 0.511689,
  0.200746, 0.507361,
  0.203333, 0.517193,
  0.205632, 0.520342,
  0.207356, 0.514439,
  0.210230, 0.506970,
  0.214829, 0.510510,
  0.219139, 0.513655,
  0.224600, 0.514050,
  0.235523, 0.515620,
  0.236959, 0.519554,
  0.243858, 0.525454,
  0.251619, 0.530567,
  0.256792, 0.538036,
  0.260529, 0.541183,
  0.265989, 0.543545,
  0.269151, 0.544332,
  0.276050, 0.554162,
  0.278061, 0.566356,
  0.273175, 0.575008,
  0.280073, 0.572648,
  0.283523, 0.572648,
  0.285248, 0.573827,
  0.284959, 0.576581,
  0.287834, 0.575008,
  0.293294, 0.577760,
  0.298755, 0.584054,
  0.302492, 0.583661,
  0.308529, 0.584840,
  0.313416, 0.586020,
  0.322325, 0.591525,
  0.326637, 0.595458,
  0.329511, 0.597426,
  0.333535, 0.602142,
  0.333821, 0.607254,
  0.329511, 0.619053,
  0.326637, 0.624559,
  0.324337, 0.628099,
  0.322038, 0.639503,
  0.320889, 0.644616,
  0.319740, 0.658776,
  0.318590, 0.669001,
  0.317440, 0.676079,
  0.313704, 0.688665,
  0.310830, 0.691811,
  0.305656, 0.694958,
  0.295884, 0.698498,
  0.290423, 0.705968,
  0.290423, 0.717376,
  0.286112, 0.727992,
  0.284676, 0.735463,
  0.283814, 0.739396,
  0.278927, 0.747263,
  0.277777, 0.750801,
  0.273753, 0.754732,
  0.272890, 0.755913,
  0.267716, 0.754732,
  0.262255, 0.751980,
  0.265130, 0.759060,
  0.270878, 0.765354,
  0.268866, 0.769680,
  0.268003, 0.771254,
  0.263979, 0.777548,
  0.261392, 0.778727,
  0.256794, 0.776760,
  0.255356, 0.780298,
  0.254781, 0.790132,
  0.249894, 0.790132,
  0.249319, 0.788164,
  0.247308, 0.788951,
  0.247883, 0.794456,
  0.250469, 0.798388,
  0.251045, 0.805074,
  0.249608, 0.808221,
  0.247308, 0.812941,
  0.247883, 0.818839,
  0.252482, 0.822773,
  0.252195, 0.831818,
  0.249321, 0.840469,
  0.250184, 0.847156,
  0.250759, 0.849516,
  0.245025, 0.849464,
  0.170858, 0.529384,
  0.168847, 0.525452,
  0.166547, 0.522699,
  0.161947, 0.517193,
  0.160510, 0.512869,
  0.160222, 0.501071,
  0.161085, 0.488879,
  0.160510, 0.485339,
  0.151887, 0.483373,
  0.145563, 0.484945,
  0.149300, 0.471182,
  0.152461, 0.459385,
  0.151599, 0.456240,
  0.145276, 0.455058,
  0.142402, 0.458204,
  0.137803, 0.468823,
  0.132630, 0.471970,
  0.126880, 0.471184,
  0.118258, 0.463713,
  0.116534, 0.452308,
  0.119407, 0.439330,
  0.122282, 0.428317,
  0.123719, 0.421632,
  0.130042, 0.416519,
  0.135215, 0.412193,
  0.140390, 0.410620,
  0.146138, 0.410620,
  0.150737, 0.413765,
  0.154761, 0.411013,
  0.159360, 0.405506,
  0.163671, 0.406294,
  0.167407, 0.407867,
  0.172006, 0.411798,
  0.173155, 0.414158,
  0.174306, 0.418483,
  0.175742, 0.429889,
  0.178616, 0.434608,
  0.181202, 0.430284,
  0.182065, 0.419664,
  0.180916, 0.406292,
  0.183789, 0.398426,
  0.197585, 0.385449,
  0.204195, 0.379550,
  0.206208, 0.375224,
  0.212244, 0.364999,
  0.216269, 0.359495,
  0.220005, 0.354383,
  0.226041, 0.348879,
  0.231503, 0.347303,
  0.234377, 0.339831,
  0.242425, 0.331965,
  0.250760, 0.327641,
  0.253347, 0.327641,
  0.250185, 0.333540,
  0.253634, 0.336685,
  0.260820, 0.329213,
  0.265706, 0.328425,
  0.270306, 0.325673,
  0.271168, 0.322135,
  0.268006, 0.321740,
  0.261107, 0.323313,
  0.257084, 0.318592,
  0.261682, 0.311907,
  0.251623, 0.312693,
  0.247023, 0.312693,
  0.247023, 0.308762,
  0.256509, 0.304040,
  0.274904, 0.302075,
  0.282952, 0.300895,
  0.289563, 0.294209,
  0.292437, 0.290277,
  0.293587, 0.285165,
  0.289563, 0.280838,
  0.286977, 0.281232,
  0.283815, 0.274153,
  0.282378, 0.268255,
  0.282953, 0.260388,
  0.282953, 0.255278,
  0.280080, 0.254883,
  0.268870, 0.262749,
  0.265707, 0.255278,
  0.264845, 0.250165,
  0.257372, 0.245444,
  0.250474, 0.244265,
  0.242137, 0.254883,
  0.238402, 0.260388,
  0.236966, 0.276907,
  0.228342, 0.281232,
  0.224607, 0.284772,
  0.221444, 0.296176,
  0.213971, 0.293424,
  0.214258, 0.282413,
  0.216845, 0.278873,
  0.198736, 0.273367,
  0.192701, 0.266682,
  0.192414, 0.257243,
  0.204772, 0.243873,
  0.217707, 0.236793,
  0.224318, 0.234435,
  0.225755, 0.228929,
  0.226329, 0.224208,
  0.229779, 0.227355,
  0.232079, 0.228139,
  0.238977, 0.222242,
  0.246739, 0.219490,
  0.252486, 0.212410,
  0.250186, 0.209263,
  0.245875, 0.212803,
  0.238977, 0.215948,
  0.234953, 0.216736,
  0.235528, 0.211624,
  0.233804, 0.210445,
  0.231504, 0.207300,
  0.230067, 0.200220,
  0.222594, 0.206905,
  0.211958, 0.221063,
  0.208509, 0.218702,
  0.202473, 0.215950,
  0.190114, 0.215950,
  0.184076, 0.216738,
  0.179766, 0.223425,
  0.176317, 0.219492,
  0.170569, 0.217919,
  0.165396, 0.215952,
  0.165396, 0.212412,
  0.145849, 0.211233,
  0.137515, 0.208481,
  0.124005, 0.210446,
  0.109633, 0.215952,
  0.101298, 0.214379,
  0.100435, 0.212414,
  0.097562, 0.208482,
  0.093538, 0.203370,
  0.080605, 0.203370,
  0.065371, 0.206515,
  0.053586, 0.207303,
  0.043527, 0.211234,
  0.035192, 0.212809,
  0.029155, 0.218314,
  0.031454, 0.223427,
  0.029155, 0.227358,
  0.025419, 0.225786,
  0.017370, 0.224606,
  0.013921, 0.229720,
  0.019095, 0.233652,
  0.022832, 0.230507,
  0.024844, 0.232867,
  0.019095, 0.239161,
  0.011048, 0.240340,
  0.005299, 0.242701,
  0.000700, 0.248600,
  0.000700, 0.254105,
  0.000987, 0.256857,
  0.006448, 0.260395,
  0.007597, 0.261968,
  0.008172, 0.264722,
  0.016507, 0.261968,
  0.023980, 0.258037,
  0.029154, 0.252922,
  0.030303, 0.255283,
  0.032029, 0.256067,
  0.036053, 0.253709,
  0.042664, 0.250955,
  0.046113, 0.252136,
  0.047262, 0.252136,
  0.054735, 0.253709,
  0.066232, 0.258821,
  0.069394, 0.261180,
  0.072556, 0.266292,
  0.071981, 0.285563,
  0.071406, 0.290676,
  0.073994, 0.312305,
  0.070257, 0.313878,
  0.068820, 0.320956,
  0.064509, 0.330001,
  0.058473, 0.339831,
  0.053874, 0.350058,
  0.050712, 0.357530,
  0.051287, 0.374048,
  0.054449, 0.386634,
  0.058473, 0.390958,
  0.062784, 0.397252,
  0.063071, 0.411412,
  0.065083, 0.419669,
  0.067382, 0.428322,
  0.069108, 0.433436,
  0.071120, 0.436974,
  0.072557, 0.441692,
  0.072845, 0.443268,
  0.071982, 0.434614,
  0.070833, 0.427929,
  0.069396, 0.419276,
  0.066810, 0.411803,
  0.065948, 0.403544,
  0.070260, 0.403544,
  0.073134, 0.412591,
  0.075145, 0.420849,
  0.079169, 0.432253,
  0.081755, 0.436186,
  0.086068, 0.447198,
  0.089229, 0.454669,
  0.088941, 0.460568,
  0.089515, 0.468041,
  0.093827, 0.471186,
  0.098714, 0.475905,
  0.103025, 0.479443,
  0.107335, 0.482983,
  0.111936, 0.484558,
  0.115096, 0.485737,
  0.118546, 0.486130,
  0.121707, 0.484558,
  0.125444, 0.484949,
  0.128893, 0.488491,
  0.132055, 0.492029,
  0.136366, 0.495174,
  0.140677, 0.496747,
  0.143552, 0.497141,
  0.146425, 0.497929,
  0.149298, 0.503040,
  0.152460, 0.508153,
  0.153322, 0.513658,
  0.155335, 0.518379,
  0.159357, 0.523095,
  0.163668, 0.524276,
  0.167118, 0.527816,
  0.170757, 0.529389,
  0.262981, 0.164030,
  0.280227, 0.161081,
  0.289281, 0.158131,
  0.298334, 0.152823,
  0.309545, 0.150463,
  0.320755, 0.152823,
  0.331963, 0.154003,
  0.342742, 0.155182,
  0.354814, 0.155182,
  0.368610, 0.153412,
  0.379390, 0.156361,
  0.392755, 0.156361,
  0.406983, 0.156361,
  0.418193, 0.155773,
  0.425521, 0.155182,
  0.432850, 0.158722,
  0.433282, 0.163441,
  0.424658, 0.171110,
  0.419053, 0.184089,
  0.407412, 0.188808,
  0.406982, 0.201196,
  0.403101, 0.208274,
  0.394479, 0.210044,
  0.381545, 0.213584,
  0.371628, 0.217713,
  0.365592, 0.224203,
  0.351795, 0.231871,
  0.348346, 0.240132,
  0.338860, 0.251932,
  0.332824, 0.246622,
  0.328081, 0.237183,
  0.325925, 0.225384,
  0.325925, 0.214766,
  0.334117, 0.214177,
  0.336705, 0.208867,
  0.334979, 0.198837,
  0.332393, 0.188810,
  0.327219, 0.182321,
  0.320321, 0.176422,
  0.312559, 0.174652,
  0.311265, 0.169932,
  0.322907, 0.170521,
  0.328081, 0.165213,
  0.325925, 0.161083,
  0.311267, 0.165802,
  0.300919, 0.168751,
  0.288848, 0.174061,
  0.286260, 0.178190,
  0.282381, 0.178190,
  0.275052, 0.180551,
  0.269878, 0.177601,
  0.268154, 0.176422,
  0.260393, 0.172291,
  0.260824, 0.167572,
  0.262969, 0.164030,
  0.253493, 0.190576,
  0.247457, 0.199426,
  0.252200, 0.205916,
  0.261685, 0.202966,
  0.272894, 0.209456,
  0.275480, 0.223614,
  0.269444, 0.228924,
  0.266426, 0.236002,
  0.277636, 0.243672,
  0.284102, 0.240134,
  0.284964, 0.232464,
  0.289708, 0.231284,
  0.287121, 0.222436,
  0.292294, 0.225385,
  0.294019, 0.220666,
  0.292294, 0.212997,
  0.286690, 0.205329,
  0.284964, 0.198250,
  0.275479, 0.194120,
  0.265131, 0.191761,
  0.256509, 0.194710,
  0.253508, 0.190576,
  0.157347, 0.197656,
  0.169420, 0.188218,
  0.178904, 0.189988,
  0.187958, 0.198836,
  0.196150, 0.197656,
  0.206066, 0.197068,
  0.203479, 0.208276,
  0.191406, 0.211816,
  0.184076, 0.210637,
  0.178471, 0.211816,
  0.172868, 0.211226,
  0.169849, 0.204736,
  0.168986, 0.200017,
  0.162519, 0.198837,
  0.157360, 0.197656
};

bool get_shape_data(ShapeData& dst, ShapeKind kind) {
  switch (kind) {
    case ShapeKind::kButterfly:
      dst.size = sizeof(butterfly_commands) - 1;
      dst.commands = butterfly_commands;
      dst.vertices = reinterpret_cast<const BLPoint*>(butterfly_vertices);
      return true;

    case ShapeKind::kFish:
      dst.size = sizeof(fish_commands) - 1;
      dst.commands = fish_commands;
      dst.vertices = reinterpret_cast<const BLPoint*>(fish_vertices);
      return true;

    case ShapeKind::kDragon:
      dst.size = sizeof(dragon_commands) - 1;
      dst.commands = dragon_commands;
      dst.vertices = reinterpret_cast<const BLPoint*>(dragon_vertices);
      return true;

    case ShapeKind::kWorld:
      dst.size = sizeof(world_commands) - 1;
      dst.commands = world_commands;
      dst.vertices = reinterpret_cast<const BLPoint*>(world_vertices);
      return true;

    default:
      dst = ShapeData{};
      return false;
  }
}

} // {blbench}
